/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// This class' package. Duh ;^)
package kirk.util;

// Import the famous StringTokenizer.
import java.util.StringTokenizer;

/**
 * <p>This class represents a wildcard.</p>
 * 
 * <p>You construct this class by supplying the wildcard <code>String</code>, and
 * then you can use it to check whether <code>String</code>s comply to the wildcard.
 * </p>
 * 
 * <p>The default values for single character checking and any amount of character
 * checking are ? and *, but you can specify your own when constructing.</p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 1.00
 */
public class Wildcard {
	/**
	 * Constructs a new <code>Wildcard</code> with the specified wildcard, and the
	 * specified character values.
	 * 
	 * @param wildcard The wildcard that strings that are tested with this object
	 * must comply with to pass.
	 * @param single The character that specifies a single character in a wildcard.
	 * @param anyamount The character that specifies any amount of characters in a
	 * wildcard.
	 */
	public Wildcard(String wildcard, char single, char anyamount) {
		// Set the wildcard.
		this.wildcard = wildcard;
		
		// Set the single character.
		this.single = single;
		
		// Set the any amount character.
		this.anyamount = anyamount;
	}

	/**
	 * Constructs a new <code>Wildcard</code> with the specified wildcard, and the
	 * default character values.
	 * 
	 * @param wildcard The wildcard that strings that are tested with this object
	 * must comply with to pass.
	 */	
	public Wildcard(String wildcard) {
		// Overload the previous constructor, using default values for single and any
		// amount.
		this(wildcard, '?', '*');
	}
	
	
	/**
	 * Returns whether the passed line complies with this <code>Wildcard</code>.
	 * 
	 * @param line The line to be checked.
	 * @return true when <b>line</b> complies with this <code>Wildcard</code>.
	 */
	public boolean test(String line) {
		// Tokenize the wildcard by constructing the famous StringTokenizer around
		// it. We want the delimiters included.
		// Startng index is 0. 		
		// Call testSingle and return it's result. The method is, together with
		// testAnyAmount, beautifully dual recursive... I love recursion.
		return testSingle(new StringTokenizer(getWildcard(), "" + single + anyamount, true), line, 0);
	}
	

	/**
	 * <p>Returns whether the passed line complies to the passed wildcard indicated
	 * by tokenizer, up to the point where an anyamount token appears in the
	 * wildcard.</p>
	 * 
	 * <p>This method assumes that when it is called, no any amount of characters are
	 * allowed before the next token. As soon as this is allowed, this method passes
	 * control back to the calling method, which assumes that there are any amount of
	 * characters allowed before the next token (or is the starting method).</p>
	 * 
	 * <p>This method works recursively, for as long as it is not in anyamount
	 * "mode".</p>
	 * 
	 * @param tokenizer The wildcard in a <code>StringTokenizer</code>.
	 * @param line The line to be tested.
	 * @param index The current position into line.
	 * @return Whether the passed line complies to the passed wildcard indicated by
	 * tokenizer.
	 */	
	protected boolean testSingle(StringTokenizer tokenizer, String line, int index) {
		// Check to see if there are characters at the current position. We do this
		// by first checking if there are more tokens in wildcard. If there are, we
		// need to do further checking.
		if(tokenizer.hasMoreTokens()) {
			// There are more tokens.
			// The line now starts out of anyamount "mode".
			// Retrieve the next token.
			String token = tokenizer.nextToken();
			
			// Check what this next token is. There are 3 options.
			// 1: single
			// This means that the next character in line is a single character with
			// any value.
			if(token.equals("" + single)) {
				// Check to see if there is a character at the current position.
				// We do this by checking if the length of line allows for at least
				// one more character, going from the position specified by index.
				if(index < line.length()) {
					// Line complies to this part of wildcard, so move the position
					// pointer one to the right. 
					index += 1;
				
					// Call this method again (recursively). 
					return testSingle(tokenizer, line, index);
				} 
				// If there are no more characters, then line fails the test.
				// We indicate this by returning false.
				else  return false;
			}
			// 2: any amount
			// This means that the next token is any amount of characters with any
			// value.
			else if(token.equals("" + anyamount)) {
				// Since the last token was single, this means we are now going into
				// any amount "mode".
				// Call the anyamount method. 
				return testAnyAmount(tokenizer, line, index);
			}
			// 3: <string>
			// Indicates that the specific String must be in line at the current
			// position.
			else {
				// Check to see if the current token is in line at the current
				// position. We do this by checking if the substring at the current
				// position starts with the token.
				if(line.startsWith(token, index)) {
					// If it does, we increase the position pointer with the length
					// of the token.
					index += token.length();
					
					// line passed this part of the test. Since we encountered a
					// normal token we are out of any amount "mode". So, we return
					// true to end this recursion and indicate that the calling
					// method can continue.
					return true;
				}
				// If it doesn't, line fails the test and we return false to indicate
				// this.
				else  return false;
			}
		}
		
		// There aren't anymore tokens, so line passed the test.
		// We return true to indicate this.
		else  return true;
	}

	/**
	 * <p>Returns whether the passed line complies to the passed wildcard indicated
	 * by tokenizer, up to the point where a string token appears in the wildcard.
	 * </p>
	 * 
	 * <p>This method assumes that when it is called, any amount of characters are
	 * allowed before the next token. As soon as this isn't allowed anymore, this
	 * method passes control back to the calling method, which assumes that there are
	 * no characters allowed before the next token.</p>
	 * 
	 * <p>This method works recursively, for as long as it is in anyamount "mode".
	 * </p>
	 * 
	 * @param tokenizer The wildcard in a <code>StringTokenizer</code>.
	 * @param line The line to be tested.
	 * @param index The current position into line.
	 * @return Whether the passed line complies to the passed wildcard indicated by
	 * tokenizer.
	 */	
	protected boolean testAnyAmount(StringTokenizer tokenizer, String line, int index) {
		// Check to see if there are any number of characters at the current
		// position. We do this by first checking if there are more tokens in
		// wildcard. If there are, we need to do further checking.
		if(tokenizer.hasMoreTokens()) {
			// There are more tokens.
			// The line now starts in anyamount "mode".
			// Retrieve the next token.
			String token = tokenizer.nextToken();
			
			// Check what this next token is. Again, there are 3 options.
			// 1: single
			// This means that the next token is a single character with any value.
			if(token.equals("" + single)) {
				// There has to be at least one character at the current position, so we
				// check to see if there is a character at the current position, and
				// then we proceed like with anyamount.
				// We do this by checking if the length of line allows for at least
				// one more character, going from the position specified by index.
				if(index < line.length()) {
					// Line complies to this part of wildcard, so move the position
					// pointer one to the right. 
					index += 1;
				
					// Call this method again (recursively). 
					return testAnyAmount(tokenizer, line, index);	
				} 
				// If there are no more characters, then line fails the test.
				// We indicate this by returning false.
				else  return false;
			}
			// 2: any amount
			// This means that the next token is any amount of characters with any
			// value.
			else if(token.equals("" + anyamount)) {
				// Since the last token was anyamount, this is useless, and can be
				// skipped.
				// Call this method again (recursively). 
				return testAnyAmount(tokenizer, line, index);
			}
			// 3: <string>
			// Indicates that the specific String must be somewhere in line after the
			// current position.
			else {
				// We look into line, starting from the current position,
				// and check if the token is somewhere there.
				// If it is, we continue the test, and set the position pointer
				// to the index of the first occurance of the next token plus the
				// token's length. That's right.
				// The index of the token in line.
				int tokenindex = line.indexOf(token, index);
				
				// Check if the index is not negative (if it is the token wasn't
				// found). 
				if(tokenindex >= 0) {
					// Set the new index into line.
					index = tokenindex + token.length();
	
					// line passed this part of the test. Since we encountered a
					// normal token we are out of any amount "mode". So, we return
					// true to end this recursion and indicate that the calling
					// method can continue.
					return true;
				}
				// If the index is negative, then the token wasn't found.
				// Return false to indicate that line failed the test.
				else  return false;
			}
		}
		
		// There aren't anymore tokens, so line passed the test.
		// We return true to indicate this.
		else  return true;
	}
	
	
	/**
	 * Returns the wildcard this object represents.
	 * 
	 * @return The wildcard this object represents.
	 */
	public String getWildcard() {
		// Return the wildcard this object represents.
		return wildcard;
	}
	
	
	/**
	 * The wildcard this class represents.
	 */
	protected String wildcard;
	
	/**
	 * The character that indicates any single character.
	 * Its standard value is '?'.
	 */
	protected char single;
	
	/**
	 * The character that indicates any amount of characters.
	 * Its standard value is '*'.
	 */
	protected char anyamount;
}