/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// This is an SDL class.
package kirk.io.sdl;

// Import the io classes.
import java.io.*;

/**
 * <p>This class writes an SDL description indicated by an <code>SDLDocument</code>
 * to an <code>OutputStream</code>, using UTF-16. To make things as easy as possible,
 * all you have to do is:</p>
 * 
 * <p><code>SDLWriter.writeSDL(&lt;SDLDocument&gt;, &lt;OutputStream&gt;);</code>
 * (when writing to an <code>OutputStream</code>)</p>
 * 
 * <p>or</p>
 * 
 * <p><code>SDLWriter.writeSDLFile(&lt;SDLDocument&gt;, &lt;filename&gt;);</code>
 * (when writing to a file).</p>
 * 
 * <p>This also means you cannot instantiate this class yourself. If for some reason
 * you need to do this, extend this class. The constructor is protected.</p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 1
 * @see kirk.io.sdl.SDLDocument
 */
public class SDLWriter {
	/**
	 * Constructs a new <code>SDLWriter</code> with the given output source to write
	 * to. If you wish to write to a file for example, supply a
	 * <code>new FileOutputStream("filename")</code>. The output source will write
	 * raw byte data to the stream using UTF-16 encoding.
	 * 
	 * @param out The <code>OutputStream</code> to be written to.
	 * @throws IOException When an IO exception occurs. Mmmyeah.
	 */
	protected SDLWriter(OutputStream out) throws IOException {
		// Set the Writer. It is a BufferedWriter which takes its data from an
		// OutputStreamWriter, which uses UTF-16 as charset. So the outgoing data is
		// written in UTF-16 format.
		// If for some reason the JVM we're running on doesn't recognize UTF-16, an
		// UnsupportedEncodingException will be thrown. This is however an
		// IOException, so we won't specifically throw it.
		writer = new BufferedWriter(new OutputStreamWriter(out, "UTF-16"));
	}
	
	
	/**
	 * Writes the given <code>SDLDocument</code> to the output stream.
	 * 
	 * @param document The <code>SDLDocument</code> to write.
	 * @throws IOException When IO errors occur.
	 * @throws SDLWriteException When SDL errors occur.
	 */
	protected void writeSDLDocument(SDLDocument document) throws IOException, SDLWriteException  {
		// In order to write an SDLDocument, we need to write its root. All its
		// elements and data will then be written to the output recursively.
		// Write the root element.
		writeElement(document.getRoot());
	}
	
	/**
	 * Writes the given <code>SDLElement</code> to the output stream.
	 * 
	 * @param element The <code>SDLElement</code> to write.
	 * @throws IOException When IO errors occur.
	 * @throws SDLWriteException When SDL errors occur.
	 */
	protected void writeElement(SDLElement element) throws IOException, SDLWriteException {
		// In order to write an SDLElement, we need to distinguish between data and
		// normal elements. After that, we write the correct things depending on
		// that. Data will be written normally, and elements will be written
		// recursively again.
		// Check the type of element, and call the correct writing method.
		if(element instanceof SDLNormalElement)  writeNormalElement((SDLNormalElement) element); 
		else if(element instanceof SDLDataElement)  writeDataElement((SDLDataElement) element);
		else {
			// Uhm... Ok?
			throw getException("SDLElement is of an unknown type.");
		}
	}
	
	
	/**
	 * Writes the given <code>SDLNormalElement</code> to the output stream.
	 * 
	 * @param element The <code>SDLNormalElement</code> to write.
	 * @throws IOException When IO errors occur.
	 * @throws SDLWriteException When SDL errors occur.
	 */
	protected void writeNormalElement(SDLNormalElement element) throws IOException, SDLWriteException {
		// In order to write an SDLNormalElement we need to write its start tag,
		// then its name, then its value (which are elements, thus recursively), and
		// then its end tag.
		// Write the start tag.
		writeStartTag(SDLNormalElement.IDENTIFIER);
		
		// Write the name.
		writeFullTag("name", element.getName());
		
		// Write the value start tag.
		writeStartTag("value");
		
		// Go through the values.
		for(int index = 0; index < element.getNumberOfValues(); index++) {
			// Write the current value to the stream.
			writeElement(element.getValue(index));
		}
		
		// Write the value end tag.
		writeEndTag("value");
		
		// Write the end tag.
		writeEndTag(SDLNormalElement.IDENTIFIER);
	}
	
	/**
	 * Writes the given <code>SDLDataElement</code> to the output stream.
	 * 
	 * @param element The <code>SDLDataElement</code> to write.
	 * @throws IOException When IO errors occur.
	 */
	protected void writeDataElement(SDLDataElement element) throws IOException {
		// In order to write an SDLDataElement we need to write its start tag,
		// then its name, then its value, and then its end tag.
		// Write the start tag.
		writeStartTag(SDLDataElement.IDENTIFIER);
		
		// Write the name.
		writeFullTag("name", element.getName());
		
		// Write the value.
		writeFullTag("value", element.getValue());
		
		// Write the end tag.
		writeEndTag(SDLDataElement.IDENTIFIER);
	}
	
	
	/**
	 * Writes the given tagname as start tag to the stream.
	 * 
	 * @param tagname The name of the start tag to be written.
	 * @throws IOException When IO errors occur.
	 */
	protected void writeStartTag(String tagname) throws IOException {
		// Create and write the start tag.
		write("<" + tagname + ">");
	}
	
	/**
	 * Writes the given tagname as end tag to the stream.
	 * 
	 * @param tagname The name of the end tag to be written.
	 * @throws IOException When IO errors occur.
	 */
	protected void writeEndTag(String tagname) throws IOException {
		// Overload the start tag writer.
		writeStartTag("/" + tagname);
	}

	/**
	 * Writes a tag pair with the given tag name and tag value to the stream.
	 * 
	 * @param name The name of the tag pair to be written.
	 * @param value The value of the tag pair to be written.
	 * @throws IOException When IO errors occur.
	 */
	protected void writeFullTag(String name, String value) throws IOException {
		// First write the start tag.
		writeStartTag(name);
		
		// Then write the value.
		write(value);
		
		// Then write the end tag.
		writeEndTag(name); 
	}	
	
	
	/**
	 * Writes a string to the output stream.
	 * 
	 * @param string The <code>String</code> to be written.
	 * @throws IOException When IO errors occur.
	 */
	protected void write(String string) throws IOException {
		// Write the String to the output stream.
		writer.write(string);
		
		// Increase the position pointer.
		position += string.length();
	}
	
	
	/**
	 * Returns an <code>SDLWriteException</code> with the given message, and some
	 * other information, like the place it happened in the file.
	 * 
	 * @param error The error that occured.
	 * @return The <code>SDLWriteException</code> that signifies the given error.
	 */
	protected SDLWriteException getException(String error) {
		// Construct and throw the exception.
		return new SDLWriteException("There is an error in the internal SDL description while writing on position " + position + ": " + error);
	}
	
	
	/**
	 * The finalize method. It closes the stream.
	 * 
	 * @see java.lang.Object#finalize()
	 */
	public void finalize() {
		// Catch errors.
		try {
			// Close the stream.
			close();
		}
		catch(IOException e) {
			// Do t3h nothing.
		}
	}
	
	
	/**
	 * Closes the underlying stream.
	 * 
	 * @throws IOException When IO errors occur.
	 */
	protected void close() throws IOException {
		// Close the stream.
		writer.close();
	}
	
	
	/**
	 * Writes the given <code>SDLDocument</code> to the given
	 * <code>OutputStream</code>.
	 * 
	 * @param document The <code>SDLDocument</code> that should be written.
	 * @param out The <code>OutputStream</code> that it should be written to.
	 * @throws IOException When IO errors occur.
	 * @throws SDLWriteException When SDL errors occur.
	 */
	public static void writeSDL(SDLDocument document, OutputStream out) throws IOException, SDLWriteException {
		// Construct a new SDLWriter.
		SDLWriter sdlwriter = new SDLWriter(out);
		
		// Ask it to write the given SDLDocument to its outputstream.
		sdlwriter.writeSDLDocument(document);
		
		// Close the stream.
		sdlwriter.close();
	}
	
	/**
	 * Writes the given <code>SDLDocument</code> to file with the given filename.
	 * 
	 * @param document The <code>SDLDocument</code> that should be written.
	 * @param filename The name of the file it should be written to.
	 * @throws IOException When IO errors occur.
	 * @throws SDLWriteException When SDL errors occur.
	 */
	public static void writeSDLFile(SDLDocument document, String filename) throws IOException, SDLWriteException {
		// Construct a FileOutputStream and a overload it to the static method above.
		writeSDL(document, new FileOutputStream(filename));
	}
	
	
	/**
	 * The <code>Writer</code> the SDL document is written to.
	 */
	protected BufferedWriter writer;
	
	/**
	 * The position in the file. This is used to indicate where errors occur.
	 */
	protected int position = 1;
}