/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// Do t3h monkah!
package kirk.io.sdl;

// Import the io classes.
import java.io.*;

/**
 * <p>This class is able to read SDL files into <code>SDLDocument</code>s. To make
 * things as easy as possible, all you have to do is:</p>
 * 
 * <p><code>SDLDocument sdldocument = SDLReader.readSDL(&lt;InputStream&gt;);</code>
 * (when reading from an <code>InputStream</code>)</p>
 * 
 * <p>or</p>
 * 
 * <p><code>SDLDocument sdldocument = SDLReader.readSDLFile(&lt;filename&gt;);</code>
 * (when reading from a file).</p>
 * 
 * <p>This also means you cannot instantiate this class yourself. If for some reason
 * you need to do this, extend this class. The constructor is protected.</p>
 * 
 * <p>Note that the SDL data must be encoded with UTF-16.</p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 2.00
 * @see kirk.io.sdl.SDLDocument
 */
public class SDLReader {
	/**
	 * Constructs a new <code>SDLReader</code> with the given input source to read
	 * from. If you wish to read from a file for example, supply a
	 * <code>new FileInputStream("filename")</code>. The input source should provide
	 * raw byte data, which is converted to characters using UTF-16 encoding.
	 * 
	 * @param in The <code>InputStream</code> to be read from.
	 * @throws IOException When an IO exception occurs. Mmmyeah.
	 */
	protected SDLReader(InputStream in) throws IOException {
		// Set the Reader. It is a BufferedReader which takes its data from an
		// InputStreamReader, which uses UTF-16 as charset. So the incoming data must
		// be in UTF-16 format, otherwise it will not be read right.
		// If for some reason the JVM we're running on doesn't recognize UTF-16, an
		// UnsupportedEncodingException will be thrown. This is however an
		// IOException, so we won't specifically throw it. 
		reader = new BufferedReader(new InputStreamReader(in, "UTF-16"));
	}


	/**
	 * Reads the SDL document from the given source and returns it as an
	 * <code>SDLDocument</code> object.
	 * 
	 * @return An <code>SDLDocument</code> tree with all the SDL data from the file.
	 * @throws IOException When IO errors occur (duh).
	 * @throws SDLReadException When the SDL is not correct.
	 */
	protected SDLDocument readSDLDocument() throws IOException, SDLReadException {
		// Let t3h reading begin!
		// This is how it works:
		// First we read an element. Then we read the body of that element, and then
		// we read the next element. Since an SDL file has only one root element, we
		// can start this process once and then, after that, the file should be
		// entirely read in.
		// Read the root element (which will make the file recusively be read in).
		SDLElement root = readElement();
		
		// Now the stream should have no more characters.
		if(reader.read() == -1) {
			// Close the stream.
			reader.close();
			
			// Return a document with root as root (:P).
			return new SDLDocument(root);
		}
		else {
			// T3h suxx0r. The file stream provided a correct SDL description, but
			// there's more. That is not allowed.
			throw getException("End of file expected.");
		}
	}
	
	/**
	 * Reads the element at the current position in the file, and the contained
	 * value. Then it returns it as an <code>SDLElement</code>. The current position
	 * in the file has to be followed by &lt;element&gt; or &lt;data&gt;.
	 * 
	 * @return The element at the current file position.
	 * @throws IOException When IO errors occur (duh).
	 * @throws SDLReadException When the SDL is not correct.
	 */
	protected SDLElement readElement() throws IOException, SDLReadException {
		// The element at the current position.
		SDLElement element;
		
		// The type of the current element.
		String type;
		
		// Check for the element or data tag.
		if(checkStartTag(SDLNormalElement.IDENTIFIER)) {
			// Set the type.
			type = SDLNormalElement.IDENTIFIER;
			
			// Read in a normal element.
			element = readNormalElement(); 
		} 
		else if(checkStartTag(SDLDataElement.IDENTIFIER)) {
			// Set the type.
			type = SDLDataElement.IDENTIFIER;
			
			// Read in a data element.
			element = readDataElement();
		}
		else {
			// The element tag is not there.
			throw getException("<" + SDLNormalElement.IDENTIFIER + "> or <" + SDLDataElement.IDENTIFIER + "> expected.");
		}
		
		// Check whether the element is closed correctly.
		if(checkEndTag(type)) {
			// This element is correct. Return it.
			return element;
		}
		else {
			// This element is not correctly closed!
			throw getException("<" + type + "> does not have a closing tag.");
		}
	}
	
	/**
	 * <p>Reads the normal element at the current position in the file, and the
	 * contained value. Then it returns it as an <code>SDLDNormalElement</code>. The
	 * current position in the file has NOT to be followed by &lt;element&gt;. This
	 * has already been made sure in <code>readElement()</code>.</p>
	 * 
	 * <p>This method will only read until the end of the value.
	 * <code>readElement()</code> will then check for the end tag.</p>
	 * 
	 * @return The normal element at the current file position.
	 * @throws IOException When IO errors occur (duh).
	 * @throws SDLReadException When the SDL is not correct.
	 */
	protected SDLNormalElement readNormalElement() throws IOException, SDLReadException {
		// <element> has already been read, so the next thing we expect is the name
		// tag pair.
		String name = readTagValue("name");
		
		// The parent (current) element.
		SDLNormalElement parent = new SDLNormalElement(name);
		
		// After that, we expect the value tag pair.
		if(checkStartTag("value")) {
			// Now we need to read elements until we encounter the end tag.
			while(!checkEndTag("value")) {
				// Read the next (child) element and add it to the parent element.
				parent.addValue(readElement());
			}
			
			// The value end tag was apparently read. Return the parent element.
			return parent;
		}
		else {
			// No value opening tag.
			throw getException("Opening tag <value> expected.");
		}
	}
	
	/**
	 * <p>Reads the data element at the current position in the file, and the
	 * contained value. Then it returns it as an <code>SDLDataElement</code>. The
	 * current position in the file has NOT to be followed by &lt;data&gt;. This has
	 * already been made sure in <code>readElement()</code>.</p>
	 * 
	 * </p>This method will only read until the end of the value.
	 * <code>readElement()</code> will then check for the end tag.</p>
	 * 
	 * @return The data element at the current file position.
	 * @throws IOException When IO errors occur (duh).
	 * @throws SDLReadException When the SDL is not correct.
	 */
	protected SDLDataElement readDataElement() throws IOException, SDLReadException {
		// <data> has already been read, so the next thing we expect is the name tag
		// pair. After that, we expect the value tag pair.
		// So, read those in that order from the description, and use it to construct
		// a data element. And return that, of course.
		return new SDLDataElement(readTagValue("name"), readTagValue("value"));
	}
	
	
	/**
	 * Return the value inbetween the tag pair (start and end tag) with the given
	 * name. The start tag of the given tag pair must be at the position in the file.
	 * 
	 * @param name The name of the tag.
	 * @return The value of the tag pair.
	 * @throws IOException When IO errors occur (duh).
	 * @throws SDLReadException When the SDL is not correct.
	 */
	protected String readTagValue(String name) throws IOException, SDLReadException {
		// First a name start tag needs to be present.
		if(checkStartTag(name)) {
			// The StringBuffer that contains the value.
			StringBuffer value = new StringBuffer();
			
			// Now we need to read until we encounter the end tag.
			while(!checkEndTag(name)) {
				// Read the next character and put it at the end of the value.
				value.append(read());
			}
			
			// The end tag was apparently read. Return the read value as a String.
			return value.toString();
		}
		else {
			// No opening tag.
			throw getException("Opening tag <" + name + "> expected.");
		}
	}
	
	
	/**
	 * Reads characters from the current position in the reader until as many as the
	 * length of the start tag of the given tag name are read. Then, this string is
	 * compared to the start tag of the given tag name and returned is whether it is
	 * the same. If it is the same, the stream position will be advanced to the point
	 * after the read tag.
	 * 
	 * @param tagname The tag name that the start tag that should be read of.
	 * @return Whether the given tag was read.
	 * @throws IOException When IO errors occur (duh).
	 * @throws SDLReadException When the end of the stream is reached during reading.
	 */
	protected boolean checkStartTag(String tagname) throws IOException, SDLReadException {
		// The read "tag".
		StringBuffer readtag = new StringBuffer();
		
		// Add the enclosing brackets to the tag.
		String tag = "<" + tagname + ">";
		
		// Mark the stream. This is always possible with a BufferedReader.
		// Make the readahead limit equal to the length of tag.
		reader.mark(tag.length());
		
		// Read the "tag" at the current position in the file.
		// So we read as many characters as the tag would be from the stream (which
		// is the length of the tag).
		for(int count = 0; count < tag.length(); count++) {
			// Add the read character to the tag.
			readtag.append(read());
		}
		
		// Compare the given tag with the read tag.
		if(tag.equals(readtag.toString())) {
			// They are the same, so we return true.
			return true;
		}
		else {
			// They are not the same, so we reset the stream and position pointer.
			reader.reset();
			position -= tag.length();

			// Return false to indicate that the tag is not there.
			return false;
		}
	}
	
	/**
	 * Reads characters from the current position in the reader until as many as the
	 * length of the end tag of the given tag name are read. Then, this string is
	 * compared to the end tag of the given tag name and returned is whether it is
	 * the same. If it is the same, the stream position will be advanced to the point
	 * after the read tag.
	 * 
	 * @param tagname The tag name that the end tag that should be read of.
	 * @return Whether the given tag was read.
	 * @throws IOException When IO errors occur (duh).
	 * @throws SDLReadException When the end of the stream is reached during reading.
	 */
	protected boolean checkEndTag(String tagname) throws IOException, SDLReadException {
		// Add a / in front of the given tag.
		tagname = "/" + tagname;
		
		// Compare the two tags and return whether they are equal. We reuse the
		// readStartTag() code for this :)
		return checkStartTag(tagname);
	}
	
	
	/**
	 * Reads and return the next character in the stream.
	 * 
	 * @return The next character in the stream.
	 * @throws IOException When IO errors occur (duh).
	 * @throws SDLReadException When an end of file is reached.
	 */
	protected char read() throws IOException, SDLReadException {
		// Read an int from the stream.
		int readint = reader.read();
		
		// Increase the position in the stream.
		position++;
		
		// Check if the character is "legal" (not whether it represents a number over
		// 18, you pervert!).
		if(readint > 0) {
			// Cast and return the character.
			return (char) readint;
		}
		else {
			// OMG! An illegal character. The only thing that could've happened is an
			// end-of-file (-1 is returned in such a case). So...
			// Decrease the position pointer to the actual end of the stream.
			position--;
			
			// Throw the exception.
			throw getException("Preliminary end of file reached.");
		}
	}
	
	
	/**
	 * Returns an <code>SDLReadException</code> with the given message, and some
	 * other information, like the place it happened in the file.
	 * 
	 * @param error The error that occured.
	 * @return The <code>SDLReadException</code> that signifies the given error.
	 */
	protected SDLReadException getException(String error) {
		// Construct and throw the exception.
		return new SDLReadException("There is an error in your SDL description at position " + position + ": " + error);
	}
	
	
	/**
	 * The finalize method. It closes the stream.
	 * 
	 * @see java.lang.Object#finalize()
	 */
	public void finalize() {
		// Catch errors.
		try {
			// Close the stream.
			close();
		}
		catch(IOException e) {
			// Do t3h nothing.
		}
	}
	
	
	/**
	 * Closes the underlying stream.
	 * 
	 * @throws IOException When IO errors occur.
	 */
	protected void close() throws IOException {
		// Close the stream.
		reader.close();
	}
	
	
	/**
	 * Reads the given SDL source into an SDL document, and returns that. The data
	 * must have been encoded with UTF-16.
	 * 
	 * @param in The <code>InputStream</code> that provides the SDL to be read.
	 * @return The <code>SDLDocument</code> that was constructed from the data from
	 * the source.
	 * @throws IOException When IO errors occur.
	 * @throws SDLReadException When the data does not make up a correct SDL file.
	 */
	public static SDLDocument readSDL(InputStream in) throws IOException, SDLReadException {
		// Construct a new SDLReader around the source specified as in.
		SDLReader sdlreader = new SDLReader(in);
		
		// Read the SDL document.
		SDLDocument sdldocument = sdlreader.readSDLDocument();
		
		// Close the stream.
		sdlreader.close();
		
		// Return the document.
		return sdldocument;
	}
	
	/**
	 * Reads the given SDL file into an SDL document, and returns that. The data
	 * must have been encoded with UTF-16.
	 * 
	 * @param filename The SDL file to be read.
	 * @return The <code>SDLDocument</code> that was constructed from the data in the
	 * file.
	 * @throws IOException When IO errors occur.
	 * @throws SDLReadException When the file is not a correct SDL file.
	 */
	public static SDLDocument readSDLFile(String filename) throws IOException, SDLReadException {
		// Overload the static method above with a FileInputStream, and return the
		// result.
		return readSDL(new FileInputStream(filename));  
	}
	
	
	/**
	 * The <code>Reader</code> the SDL document is read from.
	 */
	protected BufferedReader reader;
	
	/**
	 * The position in the file. This is used to indicate where errors are in an SDL
	 * description.
	 */
	protected int position = 1;
}