/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// The pakakage.
package kirk.io.sdl;

// Import the Iterator stuff, which we are going to implement for
// SDLNormalElements.
import java.util.*;

/**
 * This class is an <code>Iterator</code> that iterates over an
 * <code>SDLNormalElement</code> given a certain element type filter and optionally a
 * name.
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 2.00
 */
public class SDLNormalElementIterator implements Iterator {
	/**
	 * Constructs a new <code>SDLNormalElementIterator</code> working on the given
	 * <code>SDLNormalElement</code>, filtering out all the ones with the given name,
	 * and of the given type.
	 * 
	 * @param element The <code>SDLNormalElement</code> to iterate over.
	 * @param name The name to filter on.
	 * @param elementtype The type of element to filter on. Legal are
	 * <code>SDLNormalElementIterator.BOTH</code>,
	 * <code>SDLNormalElementIterator.NORMAL_ELEMENT</code> and
	 * <code>SDLNormalElementIterator.DATA_ELEMENT</code>.
	 * @throws IllegalArgumentException When the given element is
	 * <code>null</code>, or the given type is not a legal type.
	 */
	public SDLNormalElementIterator(SDLNormalElement element, String name, int elementtype) throws IllegalArgumentException {
		// Store the element if it's not null.
		if(element != null)  this.element = element;
		else  throw new IllegalArgumentException("The element to be iterated over cannot be null.");
		
		// Store the name.
		this.name = name;

		// Store the elementtype if it's legal.
		if(elementtype >= 0 && elementtype <= 2)  this.elementtype = elementtype;
		else  throw new IllegalArgumentException("The element type is illegal.");
	}
	
	/**
	 * Constructs a new <code>SDLNormalElementIterator</code> working on the given
	 * <code>SDLNormalElement</code>, filtering out all the ones of the given type.
	 * 
	 * @param element The <code>SDLNormalElement</code> to iterate over.
	 * @param elementtype The type of element to filter on. Legal are
	 * <code>SDLNormalElementIterator.BOTH</code>,
	 * <code>SDLNormalElementIterator.NORMAL_ELEMENT</code> and
	 * <code>SDLNormalElementIterator.DATA_ELEMENT</code>.
	 * @throws IllegalArgumentException When the given element is
	 * <code>null</code>, or the given type is not a legal type.
	 */
	public SDLNormalElementIterator(SDLNormalElement element, int elementtype) throws IllegalArgumentException {
		// Overload the first constructor with a null name.
		this(element, null, elementtype);
	}
	
	/**
	 * Constructs a new <code>SDLNormalElementIterator</code> working on the given
	 * <code>SDLNormalElement</code>.
	 * 
	 * @param element The <code>SDLNormalElement</code> to iterate over.
	 * @throws IllegalArgumentException When the given element is
	 * <code>null</code>, or the given type is not a legal type.
	 */
	public SDLNormalElementIterator(SDLNormalElement element) throws IllegalArgumentException {
		// Overload the first constructor with a null name and both types.
		this(element, null, SDLNormalElementIterator.BOTH);
	}
	
	
	/**
	 * Returns whether there is a next object to iterate to.
	 * 
	 * @return Whether there is a next object to iterate to.
	 * @see java.util.Iterator#hasNext()
	 */
	public boolean hasNext() {
		// Whether a name hit was encountered.
		boolean namematch = false;
			
		// Whether a type hit was encountered.
		boolean typematch = false;
		
		// The temporary counter to look for the next object.
		int nextcounter = counter;
			
		// Go through the element until the next hit (name and type match), or until
		// we're out of objects.
		while(nextcounter < element.getNumberOfValues() - 1) {
			// Add 1 to the counter.
			nextcounter++;
				
			// The current element.
			SDLElement sdlelement = element.getValue(nextcounter);
				
			// Check whether name filter is null. If it is, don't filter on names
			// (then names always match).
			if(name == null)  namematch = true;
			// Otherwise check whether the element's name matches the name
			// filter.
			else  namematch = sdlelement.getName().equals(name);
				
			// Check whether the element's type matches the type filter.
			if(elementtype == BOTH) {
				// The element must be either a normal or a data element.
				typematch = sdlelement instanceof SDLNormalElement || sdlelement instanceof SDLDataElement;
			}
			else if(elementtype == NORMAL_ELEMENT) {
				// The element must be a normal element.
				typematch = sdlelement instanceof SDLNormalElement;
			} 
			else if(elementtype == DATA_ELEMENT) {
				// The element must be a normal element.
				typematch = sdlelement instanceof SDLDataElement;
			}
			
			// Check whether a match was found and return if there was.
			if(namematch && typematch)  return true;
		}
		
		// If we get here, there are no more objects that satisfy our demands.
		return false;
	}
	
	/**
	 * Returns the next object and iterates onto it. 
	 * 
	 * @return The next object and iterates onto it.
	 * @throws NoSuchElementException When there are no elements left to iterate to.
	 * @see java.util.Iterator#next()
	 */
	public Object next() throws NoSuchElementException {
		// Check whether there are more elements.
		if(hasNext()) {
			// Whether a name hit was encountered.
			boolean namematch = false;
			
			// Whether a type hit was encountered.
			boolean typematch = false;
			
			// Go through the element until the next hit (name and type match).
			while(!(namematch && typematch)) {
				// Add 1 to the counter.
				counter++;
				
				// The current element.
				SDLElement sdlelement = element.getValue(counter);
				
				// Check whether name filter is null. If it is, don't filter on names
				// (then names always match).
				if(name == null)  namematch = true;
				// Otherwise check whether the element's name matches the name
				// filter.
				else  namematch = sdlelement.getName().equals(name);
				
				// Check whether the element's type matches the type filter.
				if(elementtype == BOTH) {
					// The element must be either a normal or a data element.
					typematch = sdlelement instanceof SDLNormalElement || sdlelement instanceof SDLDataElement;
				}
				else if(elementtype == NORMAL_ELEMENT) {
					// The element must be a normal element.
					typematch = sdlelement instanceof SDLNormalElement;
				} 
				else if(elementtype == DATA_ELEMENT) {
					// The element must be a normal element.
					typematch = sdlelement instanceof SDLDataElement;
				}
			}
			
			// Return the object on the index indicated by counter.
			return element.getValue(counter);
		}
		// No next object. Throw an exception to indicate that.
		else  throw new NoSuchElementException("There were no more elements in the element with the name: " + element.getName());
	}
	
	/**
	 * Removes the current object. This method is not supported, as allowed by the
	 * <code>Iterator</code> interface. 
	 * 
	 * @throws UnsupportedOperationException If this method is not supported. Which
	 * it isn't.
	 * @throws IllegalStateException If this object is not in the state to do this
	 * method.
	 * @see java.util.Iterator#remove()
	 */
	public void remove() throws UnsupportedOperationException, IllegalStateException {
		// We don't support this method. We don't need this method.
		throw new UnsupportedOperationException("The remove() method of the SDLNormalElementIterator class is not supported.");
	}
	
	
	/**
	 * The element iterated over.
	 */
	protected SDLNormalElement element;
	
	/**
	 * The name filtered upon.
	 */
	protected String name;
	
	/**
	 * The internal (active) element type identifier.
	 */
	protected int elementtype;
	
	
	/**
	 * The element counter.
	 */
	protected int counter = -1;
	
	
	/**
	 * The element type identifier for both elements.
	 */
	public static final int BOTH = 0;
	
	/**
	 * The element type identifier for normal elements.
	 */
	public static final int NORMAL_ELEMENT = 1;
	
	/**
	 * The element type identifier for data elements.
	 */
	public static final int DATA_ELEMENT = 2;
}