/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// Package!
package kirk.io.sdl;

// Huummmmmm!
import java.util.ArrayList;
import java.util.Iterator;

/**
 * This class represents the normal element of an SDL document. It has a name and
 * element values.
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 2.00
 */
public class SDLNormalElement extends SDLElement {
	/**
	 * Constructs a new normal element with the given name.
	 * 
	 * @param name The name of this element.
	 */
	public SDLNormalElement(String name) {
		// Set the name in the superclass.
		super(name);
	}


	/**
	 * Returns the element at <code>index</code>.
	 * 
	 * @param index The index of the element to be returned.
	 * @return The element at <code>index</code>.
	 */
	public SDLElement getValue(int index) {
		// Return the element at the given index.
		return (SDLElement) valuelist.get(index);
	}
	
	/**
	 * Adds the given value to the list of values (at the specified index).
	 * Althought SDL doesn't differentiate on order of elements, you can use this to
	 * make the file look like you want.
	 * 
	 * @param value The value to be added.
	 * @param index The index to be added on.
	 */
	public void addValue(int index, SDLElement value) {
		// Check if the value is not null.
		if(value != null) {
			// Add the value to the list.
			valuelist.add(index, value);
		}
		// The value was null. That's too bad.
		else  throw new NullPointerException("Added values cannot be null.");
	}
	
	/**
	 * Adds the given value to the list of values.
	 * 
	 * @param value The value to be added.
	 */
	public void addValue(SDLElement value) {
		// Check if the value is not null.
		if(value != null) {
			// Add the value to the list.
			valuelist.add(value);
		}
		// The value was null. That's too bad.
		else  throw new NullPointerException("Added values cannot be null.");
	}
	
	/**
	 * Sets the given value in the list of values (at the specified index).
	 * 
	 * @param value The value to be set.
	 * @param index The index to be set on.
	 */
	public void setValue(SDLElement value, int index) {
		// Check if the value is not null.
		if(value != null) {
			// Set the value in the list.
			valuelist.set(index, value);
		}
		// The value was null. That's too bad.
		else  throw new NullPointerException("Set values cannot be null.");
	}

	/**
	 * Removes the given value from the list of values (from the specified index).
	 * 
	 * @param index The index to be removed from.
	 * @return The removed element.
	 */
	public SDLElement removeValue(int index) {
		// Remove the value from the list.
		return (SDLElement) valuelist.remove(index);
	}
	
	/**
	 * Removes the given value from the list of values.
	 * 
	 * @param value The element to be removed.
	 * @return The removed element. 
	 */
	public SDLElement removeValue(SDLElement value) {
		// Check if the value is not null.
		if(value != null) {
			// Remove the value from the list.
			if(valuelist.remove(value))  return value;
			else throw new IllegalArgumentException("The value to be removed is not present.");
		}
		// The value was null. That's too bad.
		else  throw new NullPointerException("Removed values cannot be null.");
	}


	/**
	 * Returns how many values there are in the list.
	 * 
	 * @return How many values there are in the list.
	 */
	public int getNumberOfValues() {
		// Return how many values there are in the list.
		return valuelist.size(); 
	}
		
	
	/**
	 * Returns the <code>Iterator</code> that iterates over this
	 * <code>SDLNormalElement</code>.
	 */
	public Iterator getElements() {
		// Return a new iterator. Tell it it should iterate over us and tell it to
		// return both types of element.
		return new SDLNormalElementIterator(this, SDLNormalElementIterator.BOTH);
	}
	
	/**
	 * Returns the <code>Iterator</code> that iterates over this
	 * <code>SDLNormalElement</code> given the name filter.
	 * 
	 * @param name The name to filter upon.
	 */
	public Iterator getElements(String name) {
		// Return a new iterator. Tell it it should iterate over us, and give it the
		// name filter. Also, tell it to return both types of element.
		return new SDLNormalElementIterator(this, name, SDLNormalElementIterator.BOTH);
	}
	
	/**
	 * Returns the <code>Iterator</code> that iterates over this
	 * <code>SDLNormalElement</code>. It only returns the
	 * <code>SDLNormalElement</code> objects.
	 */
	public Iterator getNormalElements() {
		// Return a new iterator. Tell it it should iterate over us and tell it to
		// return the normal type of element.
		return new SDLNormalElementIterator(this, SDLNormalElementIterator.NORMAL_ELEMENT);
	}
	
	/**
	 * Returns the <code>Iterator</code> that iterates over this
	 * <code>SDLNormalElement</code> given the name filter. It only returns the
	 * <code>SDLNormalElement</code> objects.
	 * 
	 * @param name The name to filter upon.
	 */
	public Iterator getNormalElements(String name) {
		// Return a new iterator. Tell it it should iterate over us, and give it the
		// name filter. Also, tell it to return the normal type of element.
		return new SDLNormalElementIterator(this, name, SDLNormalElementIterator.NORMAL_ELEMENT);
	}
	
	/**
	 * Returns the <code>Iterator</code> that iterates over this
	 * <code>SDLNormalElement</code>. It only returns the
	 * <code>SDLDataElement</code> objects.
	 */
	public Iterator getDataElements() {
		// Return a new iterator. Tell it it should iterate over us and tell it to
		// return the data type of element.
		return new SDLNormalElementIterator(this, SDLNormalElementIterator.DATA_ELEMENT);
	}
	
	/**
	 * Returns the <code>Iterator</code> that iterates over this
	 * <code>SDLNormalElement</code> given the name filter. It only returns the
	 * <code>SDLDataElement</code> objects.
	 * 
	 * @param name The name to filter upon.
	 */
	public Iterator getDataElements(String name) {
		// Return a new iterator. Tell it it should iterate over us, and give it the
		// name filter. Also, tell it to return the data type of element.
		return new SDLNormalElementIterator(this, name, SDLNormalElementIterator.DATA_ELEMENT);
	}
	
	
	/**
	 * Returns the <code>SDLElement</code> in this element that has the given name.
	 * 
	 * @param name The name of the element to return.
	 * @return The <code>SDLElement</code> that has the given name.
	 * @throws MultipleMatchException When more than one element has the given name.
	 * @throws NoMatchException When no element has the given name.
	 */
	public SDLElement getElement(String name) throws MultipleMatchException, NoMatchException {
		// The iterator over the value list (which filters on name).
		Iterator elements = getElements(name);
		
		// Check whether there are elements in the list.
		if(elements.hasNext()) {
			// Retrieve the first element.
			SDLElement element = (SDLElement) elements.next();
			
			// Check whether there is another element. If there is, there are
			// multiple matches and we throw an exception to indicate so. 
			if(elements.hasNext())  throw new MultipleMatchException("There were multiple matches while searching for an element with " + name + " as name.");
			// There is no other element. Return the single one we found.
			else  return element;
		}
		// There is no match. Throw an exception to indicate this.
		else  throw new NoMatchException("There was no match while searching for an element with " + name + " as name.");
	}
	
	/**
	 * Returns the <code>SDLNormalElement</code> in this element that has the given
	 * name.
	 * 
	 * @param name The name of the element to return.
	 * @return The <code>SDLNormalElement</code> that has the given name.
	 * @throws MultipleMatchException When more than one element has the given name.
	 * @throws NoMatchException When no element has the given name.
	 */
	public SDLNormalElement getNormalElement(String name) throws MultipleMatchException, NoMatchException {
		// The iterator over the value list (which filters on name).
		Iterator elements = getNormalElements(name);
		
		// Check whether there are elements in the list.
		if(elements.hasNext()) {
			// Retrieve the first element.
			SDLNormalElement element = (SDLNormalElement) elements.next();
			
			// Check whether there is another element. If there is, there are
			// multiple matches and we throw an exception to indicate so. 
			if(elements.hasNext())  throw new MultipleMatchException("There were multiple matches while searching for a normal element with " + name + " as name.");
			// There is no other element. Return the single one we found.
			else  return element;
		}
		// There is no match. Throw an exception to indicate this.
		else  throw new NoMatchException("There was no match while searching for a normal element with " + name + " as name.");
	}
	
	/**
	 * Returns the <code>SDLDataElement</code> in this element that has the given
	 * name.
	 * 
	 * @param name The name of the element to return.
	 * @return The <code>SDLDataElement</code> that has the given name.
	 * @throws MultipleMatchException When more than one element has the given name.
	 * @throws NoMatchException When no element has the given name.
	 */
	public SDLDataElement getDataElement(String name) throws MultipleMatchException, NoMatchException {
		// The iterator over the value list (which filters on name).
		Iterator elements = getDataElements(name);
		
		// Check whether there are elements in the list.
		if(elements.hasNext()) {
			// Retrieve the first element.
			SDLDataElement element = (SDLDataElement) elements.next();
			
			// Check whether there is another element. If there is, there are
			// multiple matches and we throw an exception to indicate so. 
			if(elements.hasNext())  throw new MultipleMatchException("There were multiple matches while searching for a data element with " + name + " as name.");
			// There is no other element. Return the single one we found.
			else  return element;
		}
		// There is no match. Throw an exception to indicate this.
		else  throw new NoMatchException("There was no match while searching for a data element with " + name + " as name.");
	}
	
	
	/**
	 * Returns a string representation of this <code>SDLNormalElement</code> with the
	 * given indentation.
	 * 
	 * @param indent The indentation to be used.
	 * @return A string representation of this <code>SDLNormalElement</code>.
	 */
	public String toString(String indent) {
		// Retrieve the newline character(s) for the current OS.
		String nl = System.getProperty("line.separator");
		
		// The string.
		String string =
			indent + "Normal Element:" + nl +
			indent + " Name: " + getName() + nl +
			indent + " Value: "  + nl;
		
		// Go through the list of values.
		for(int index = 0; index < getNumberOfValues(); index++) {
			string += getValue(index).toString(indent + "  ");
		}
			
		// Return t3h string.
		return string;
	}
	
	
	/**
	 * The list of values of this element.
	 */
	protected ArrayList valuelist = new ArrayList();
	
	/**
	 * The identifier of the normal element type.
	 */
	public static final String IDENTIFIER = "element";
}