/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// The package of this class.
package kirk.io.sdl;

/**
 * This class represents an SDL element. It has a name, and a value. What this value
 * actually is, is decided by the subclasses. SDL has 2 types of elements, the normal
 * element, and the data element. These are represented by 2 subclasses.
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 2.00
 */
public abstract class SDLElement {
	/**
	 * Constructs a new <code>SDLElement</code> with the given name.
	 * 
	 * @param name The name of the element.
	 */
	public SDLElement(String name) {
		// Set the name.
		setName(name);
	}
	
	
	/**
	 * Returns the name of this element.
	 * 
	 * @return The name of this element.
	 */
	public String getName() {
		// Return the name.
		return name;
	}

	/**
	 * Sets the name of this element to <code>name</code>.
	 * 
	 * @param name The name this element should have.
	 */
	public void setName(String name) {
		// Set the name.
		this.name = name;
	}
	
	
	/**
	 * Convenience method for casting this element to the normal type.
	 * This method is aimed at reducing clutter in code that needs a cast. It also
	 * allows easier chained statements. Casting is something that happens a lot when
	 * retrieving data from the SDL structure, since you need to know/have the
	 * subtype of the element.
	 * 
	 * @return This element as a normal element.
	 */
	public SDLNormalElement getAsNormalElement() {
		// Return this element as a normal element.
		return (SDLNormalElement) this;
	}
	
	/**
	 * Convenience method for casting this element to the data type.
	 * This method is aimed at reducing clutter in code that needs a cast. It also
	 * allows easier chained statements. Casting is something that happens a lot when
	 * retrieving data from the SDL structure, since you need to know/have the
	 * subtype of the element.
	 * 
	 * @return This element as a data element.
	 */
	public SDLDataElement getAsDataElement() {
		// Return this element as a data element.
		return (SDLDataElement) this;
	}
	
	
	/**
	 * Returns a string representation of this <code>SDLElement</code> with the given
	 * indentation.
	 * 
	 * @param indent The indentation to be used.
	 * @return A string representation of this <code>SDLElement</code>.
	 */
	public abstract String toString(String indent);
		
	
	/**
	 * The name of this element.
	 */
	protected String name;
}