/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// Zee package.
package kirk.io.sdl;

// Import the kirk utilities.
import kirk.util.*;

/**
 * This class represents the data element of an SDL document. It has a name and a
 * data (text) value.
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 2.00
 */
public class SDLDataElement extends SDLElement {
	/**
	 * Constructs a new data element with the given name and value.
	 * 
	 * @param name The name of the element.
	 * @param value The value of the element.
	 */
	public SDLDataElement(String name, String value) {
		// Supername!
		super(name);
		
		// Set the value.
		setValue(value);
	}


	/**
	 * Sets the value to the given value.
	 * 
	 * @param value The new value of this element.
	 */
	public void setValue(String value) {
		this.value = value;
	}
	
	/**
	 * Returns the value of this element.
	 * 
	 * @return The value of this element.
	 */
	public String getValue() {
		// Reetoorn eet!
		return value;
	}
	
	
	/**
	 * Returns the value of this element as a <code>String</code>.
	 * 
	 * @return The value of this element as a <code>String</code>.
	 */
	public String getValueAsString() {
		// Reetoorn eet!
		return value;
	}
	
	/**
	 * Returns the value of this element as a <code>StringBuffer</code>.
	 * 
	 * @return The value of this element as a <code>StringBuffer</code>.
	 */
	public StringBuffer getValueAsStringBuffer() {
		// Return a new StringBuffer initalised on the value.
		return new StringBuffer(value);
	}
	
	/**
	 * Returns the value of this element as a <code>Character</code>.
	 * 
	 * @return The value of this element as a <code>Character</code>.
	 * @throws CharacterFormatExeption When the value couldn't be cast to a
	 * character.
	 */
	public Character getValueAsCharacter() throws CharacterFormatException {
		// Check the length of the value.
		if(value.length() == 1) {
			// The value is a single character. Convert it and return it.
			return new Character(value.charAt(0));
		}
		else {
			// The value is NOT a single character. Throw an exception.
			throw new CharacterFormatException("The data value is not a single character: " + value);
		}
	}
	
	
	/**
	 * Returns the value of this element as a <code>Byte</code>.
	 * 
	 * @return The value of this element as a <code>Byte</code>.
	 * @throws NumberFormatExeption When the value couldn't be cast to a
	 * Byte.
	 */
	public Byte getValueAsByte() throws NumberFormatException {
		// Construct a new Byte from the String and return that.
		return new Byte(value);
	}
	
	/**
	 * Returns the value of this element as a <code>Short</code>.
	 * 
	 * @return The value of this element as a <code>Short</code>.
	 * @throws NumberFormatExeption When the value couldn't be cast to a
	 * Short.
	 */
	public Short getValueAsShort() throws NumberFormatException {
		// Construct a new Short from the String and return that.
		return new Short(value);
	}
	
	
	/**
	 * Returns the value of this element as an <code>Integer</code>.
	 * 
	 * @return The value of this element as a <code>Integer</code>.
	 * @throws NumberFormatExeption When the value couldn't be cast to a
	 * Integer.
	 */
	public Integer getValueAsInteger() throws NumberFormatException {
		// Construct a new Integer from the String and return that.
		return new Integer(value);
	}
	
	/**
	 * Returns the value of this element as an <code>Long</code>.
	 * 
	 * @return The value of this element as a <code>Long</code>.
	 * @throws NumberFormatExeption When the value couldn't be cast to a
	 * Long.
	 */
	public Long getValueAsLong() throws NumberFormatException {
		// Construct a new Long from the String and return that.
		return new Long(value);
	}
	
	/**
	 * Returns the value of this element as an <code>Float</code>.
	 * 
	 * @return The value of this element as a <code>Float</code>.
	 * @throws NumberFormatExeption When the value couldn't be cast to a
	 * Float.
	 */
	public Float getValueAsFloat() throws NumberFormatException {
		// Construct a new Float from the String and return that.
		return new Float(value);
	}

	/**
	 * Returns the value of this element as an <code>Double</code>.
	 * 
	 * @return The value of this element as a <code>Double</code>.
	 * @throws NumberFormatExeption When the value couldn't be cast to a
	 * Double.
	 */
	public Double getValueAsDouble() throws NumberFormatException {
		// Construct a new Double from the String and return that.
		return new Double(value);
	}
	
	
	/**
	 * Returns the value of this element as a <code>Boolean</code>.
	 * 
	 * @return The value of this element as a <code>Boolean</code>.
	 * @throws BooleanFormatException When the value couldn't be cast to a
	 * Boolean.
	 */
	public Boolean getValueAsBoolean() throws BooleanFormatException {
		// Check the contents of the String.
		if(value.equalsIgnoreCase("true")) {
			// The value is a true boolean. Make it and return it.
			return new Boolean(true);
		}
		else if(value.equalsIgnoreCase("false")) {
			// The value is a false boolean. Make it and return it.
			return new Boolean(false);
		}
		else {
			// The value is NOT a boolean. Throw an exception.
			throw new BooleanFormatException("The data value is not a boolean: " + value);
		}
	}
	
	
	/**
	 * Returns a string representation of this <code>SDLDataElement</code> with the
	 * given indentation.
	 * 
	 * @param indent The indentation to be used.
	 * @return A string representation of this <code>SDLDataElement</code>.
	 */
	public String toString(String indent) {
		// Retrieve the newline character(s) for the current OS.
		String nl = System.getProperty("line.separator");
		
		// The string.
		String string =
			indent + "Data Element:" + nl +
			indent + " Name: " + getName() + nl +
			indent + " Value: " + getValue() + nl;
			
		// Return t3h string.
		return string;
	}
	
	/**
	 * The data value of this element.
	 */
	protected String value;
	
	/**
	 * The identifier of the data element type.
	 */
	public static final String IDENTIFIER = "data";
}