/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// The package of this class.
package kirk.io;

// Use the IO classes of the Java 2 SDK.
import java.io.*;

// Use the StringTokenizer class of the Java 2 SDK.
import java.util.StringTokenizer;

/**
 * <p>This class reads a file token by token. Reading files token by token is only
 * useful with text based files.</p>
 * 
 * <p>Methods in this class may generate an <code>IOException</code>.</p>
 * 
 * <p>Construction is done in one of the following ways:</p>
 * <p><pre>
 * 		new TokenFileReader(String file);
 * 			Most standard. Opens the passed file or throws an IOEception. Uses the
 * 			standard delimiters (" ") and any line terminating delimiters, and no
 * 			delimiting token returning.
 * 
 * 		new TokenFileReader(String file, String delims);
 * 			Opens the passed file or throws an IOEception. Uses the passed delimiters
 * 			and any line terminating delimiters, and no delimiting token returning.
 * 
 * 		new TokenFileReader(String file, String delims, boolean delimtokens);
 * 			Opens the passed file or throws an IOEception. Uses the passed delimiters
 * 			and any line terminating delimiters, and delimiting token returning.
 * 
 * 		new TokenFileReader(String file, boolean delimtokens);
 * 			Opens the passed file or throws an IOEception. Uses the standard
 * 			delimiters (" ") and any line terminating delimiters, and delimiting
 * 			token returning.
 * </pre></p>
 * 
 * <p>Reading the tokens in a file usually goes like this:</p>
 * <p><pre>
 * 		FileTokenizer ft = new FileTokenizer("file.txt");
 * 		while(ft.fileHasMoreTokens()) {
 * 			String token = ft.nextToken();
 * 			// Do something with token.
 * 		}
 * </pre></p>
 * 
 * <p>Because of the presence of the methods that deal with the current token (see
 * below), you might want to act on the current token instead.<br>
 * Below "String token = ft.currentToken();" can be replaced with any of the other
 * token reading methods.</p>
 * <p><pre>
 * 		FileTokenizer ft = new FileTokenizer("file.txt");
 * 		while(ft.fileHasMoreTokens()) {
 * 			ft.nextToken();
 * 
 * 			String token = ft.currentToken();
 * 			// Do something with token.
 * 		}
 * </pre></p>
 * 
 * <p>Be aware that any call to currentToken() before any nextToken() is called after
 * object construction, results in a null result.</p>
 * 
 * <p>Here are the other ways to retrieve the token:</p>
 * <p><pre>
 * 		getTokenAsString() will return the same result as currenttoken().
 * 		getTokenAsByte() will return the token as an Byte object, or null if that's
 * 			not possible.
 * 		getTokenAsCharacter() will return the token as an Character object, or null
 * 			if that's not possible.
 * 		getTokenAsDouble() will return the token as an Double object, or null if
 * 			that's not possible.
 * 		getTokenAsFloat() will return the token as an Float object, or null if that's
 * 			 not possible.
 * 		getTokenAsInteger() will return the token as an Integer object, or null if
 * 			that's not possible.
 * 		getTokenAsLong() will return the token as an Long object, or null if that's
 * 			not possible.
 * 		getTokenAsShort() will return the token as an Short object, or null if that's
 * 			not possible.
 * </pre></p>
 * 
 * <p>There are also two methods for checking if there are any tokens left.</p>
 * <p><pre>
 * 		fileHasMoreTokens() will return true iff the file has more tokens.
 * 			If !fileHasMoreTokens() then nextToken() will return null. After this
 * 			call any current token accessors will return null too.
 * 		lineHasMoreTokens() will return true iff the current line in the file has
 * 			more tokens.
 * </pre></p>
 * 
 * <p>Two methods are provided for checking on lines.</p>
 * <p><pre>
 * 		firsttoken() will return true iff the current token is the first one on the
 * 			current line.
 * 		lasttoken() will return true iff the current token is the last one on the
 * 			current line.
 * </pre></p>
 * 
 * <p>Naturally a method is included to close the file: <code>close()</code>.</p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 1.20
 */
public class FileTokenizer {
	/**
	 * <p>Creates a new instance of this class with:</p>
	 * 
	 * <p><pre>file as the file this class works upon,
	 * delims as the tokenizing delimiters and
	 * delimtokens to indicate if tokens should be returned.</pre></p>
	 * 
	 * <p>The actual <code>FileReader</code> isn't used really, a much nicer class
	 * called <code>BufferedReader</code> is used.</p>
	 * 
	 * @param file The file to be tokenized.
	 * @param delims The delimiters to be used for tokenizing.
	 * @param delimtokens Whether to tokenize the delimiters too.
	 * @see java.util.StringTokenizer
	 */
	public FileTokenizer(String file, String delims, boolean delimtokens) throws IOException {
		// Construct a BufferedReader wrapping the passed file.
		reader = new BufferedReader(new FileReader(file));

		// Set the delmiters.
		delimiters = delims;

		// Set the returndelims flag.
		returndelims = delimtokens;

		// Leave the object in a consistent state. This means nextLine has to be called twice. Once to initialise the order, and another time to initialise the lines.
		nextLine();
		nextLine();
	}

	/**
	 * <p>Creates a new instance of this class with:</p>
	 * 
	 * <p><pre>file as the file this class works upon and
	 * delims as the tokenizing delimiters.<pre></p>
	 * 
	 * <p>The actual <code>FileReader</code> isn't used really, a much nicer class
	 * called <code>BufferedReader</code> is used.</p>
	 * 
	 * <p>This constructor overloads the first one.</p>
	 * 
	 * @param file The file to be tokenized.
	 * @param delims The delimiters to be used for tokenizing.
	 */
	public FileTokenizer(String file, String delims) throws IOException {
		// Overload with no delimiter returning.
		this(file, delims, false);
	}

	/**
	 * <p>Creates a new instance of this class with:</p>
	 * 
	 * <p><pre>file as the file this class works upon and
	 * delimtokens to indicate if tokens should be returned.</pre></p>
	 * 
	 * <p>The actual <code>FileReader</code> isn't used really, a much nicer class
	 * called <code>BufferedReader</code> is used.</p>
	 * 
	 * <p>This constructor overloads the first one.</p>
	 * 
	 * @param file The file to be tokenized.
	 * @param delimtokens Whether to tokoenize the delimiters too.
	 */
	public FileTokenizer(String file, boolean delimtokens) throws IOException {
		// Overload with no delimiter returning.
		this(file, " ", delimtokens);
	}

	/**
	 * <p>Creates a new instance of this class with:</p>
	 * 
	 * <p><pre>file as the file this class works upon.</pre></p>
	 * 
	 * <p>The actual <code>FileReader</code> isn't used really, a much nicer class
	 * called <code>BufferedReader</code> is used.</p>
	 * 
	 * <p>This constructor overloads the first one.</p>
	 * 
	 * @param file The file to be tokenized.
	 */
	public FileTokenizer(String file) throws IOException {
		// Overload with default delimiters.
		this(file, " ", false);
	}


	/**
	 * Closes the file reader is reading from. This should be done before the object
	 * is dismissed.
	 */
	public void close() throws IOException {
		// Close the file.
		reader.close();
	}


	/**
	 * Advances to the next line.
	 */
	protected void nextLine() throws IOException {
		// Set the current line to the next line.
		currentline = nextline;

		// Read lines to set nextline to the next line.
		do
		{
			// Set line to the next line in the file.
			nextline = reader.readLine();

			// Check if there are no more lines to be read (this is true when
			// nextline equals null).
			if(nextline == null)  break;
		}
		// Skip empty lines.
		while(nextline.equals(""));

		// Check if there actually was a line left...
		if(currentline != null) {
			// ...and construct a tokenizer over that line if there was.
			tokenizer = new StringTokenizer(currentline, delimiters, returndelims);
		}
		// No line left.
		else {
			// Set the tokenizer to null because there's nothing left to tokenize.
			tokenizer = null;
		}
	}

	/**
	 * Returns the next token in the file, or null if there is none.
	 * Also advances to the next token.
	 * 
	 * @return The next token in the file, or null if there is none.
	 */
	public String nextToken() throws IOException {
		// Check if the tokenizer exists.
		if(tokenizer != null) {
			// Check if there are any more tokens on this line.
			if(tokenizer.hasMoreTokens()) {
				// Set the current token to the next token.
				currenttoken = tokenizer.nextToken();
			}
			// No more tokens in this line.
			else {
				// Advance to the next line.
				nextLine();

				// Recursively call this method to skip lines with only delimiters
				// (in case they aren't returned).
				// As last, currenttoken will be set to the next token.
				nextToken();
			}

			// Return the next token (which is now also the current token).
			return currenttoken;
		}
		// The tokenizer doesn't exist anymore.
		else {
			// Return null, no more tokens are left to read.
			currenttoken = null;
			return currenttoken;
		}
	}

	/**
	 * Returns the current token, or null if there is none.
	 * 
	 * @return The current token, or null if there is none.
	 */
	public String currentToken() {
		// Return the current token.
		return currenttoken;
	}

	/**
	 * Returns the current token as a String (which it already is), or null if there
	 * is none.
	 * 
	 * @return The current token as a String (which it already is), or null if there
	 * is none.
	 */
	public String getTokenAsString() {
		// Return the current token.
		return currenttoken;
	}


	/**
	 * Returns the current token as a Byte, or null if there is none/it isn't a Byte.
	 * 
	 * @return The current token as a Byte, or null if there is none/it isn't a Byte.
	 */
	public Byte getTokenAsByte() {
		// Check for errors.
		try {
			// Return the current token wrapped into a Byte if possible.
			return new Byte(currenttoken);
		}
		catch(NumberFormatException e) {
			// Return null because the token isn't a Byte.
			return null;
		}
	}

	/**
	 * Returns the current token as a Character, or null if there is none/it isn't a
	 * Character.
	 * 
	 * @return The current token as a Character, or null if there is none/it isn't a
	 * Character.
	 */
	public Character getTokenAsCharacter() {
		// Check currenttoken's length.
		if(currenttoken.length() == 1) {
			// currenttoken can be seen as a char. Return the corresponding character.
			return new Character(currenttoken.charAt(0));
		}
		// currenttoken can't be a character.
		else  return null;
	}

	/**
	 * Returns the current token as a Double, or null if there is none/it isn't a
	 * Double.
	 * 
	 * @return The current token as a Double, or null if there is none/it isn't a
	 * Double.
	 */
	public Double getTokenAsDouble() {
		// Check for errors.
		try {
			// Return the current token wrapped into a Double if possible.
			return new Double(currenttoken);
		}
		catch(NumberFormatException e) {
			// Return null because the token isn't a Double.
			return null;
		}
	}

	/**
	 * Returns the current token as a Float, or null if there is none/it isn't a
	 * Float.
	 * 
	 * @return The current token as a Float, or null if there is none/it isn't a
	 * Float.
	 */
	public Float getTokenAsFloat() {
		// Check for errors.
		try {
			// Return the current token wrapped into a Float if possible.
			return new Float(currenttoken);
		}
		catch(NumberFormatException e) {
			// Return null because the token isn't a Float.
			return null;
		}
	}

	/**
	 * Returns the current token as an Integer, or null if there is none/it isn't an
	 * Integer.
	 * 
	 * @return The current token as an Integer, or null if there is none/it isn't an
	 * Integer.
	 */
	public Integer getTokenAsInteger() {
		// Check for errors.
		try {
			// Return the current token wrapped into an Integer if possible.
			return new Integer(currenttoken);
		}
		catch(NumberFormatException e) {
			// Return null because the token isn't an Integer.
			return null;
		}
	}

	/**
	 * Returns the current token as a Long, or null if there is none/it isn't a Long.
	 * 
	 * @return The current token as a Long, or null if there is none/it isn't a Long.
	 */
	public Long getTokenAsLong() {
		// Check for errors.
		try {
			// Return the current token wrapped into a Long if possible.
			return new Long(currenttoken);
		}
		catch(NumberFormatException e) {
			// Return null because the token isn't a Long.
			return null;
		}
	}

	/**
	 * Returns the current token as a Short, or null if there is none/it isn't a
	 * Short.
	 * 
	 * @return The current token as a Short, or null if there is none/it isn't a
	 * Short.
	 */
	public Short getTokenAsShort() {
		// Check for errors.
		try {
			// Return the current token wrapped into a Short if possible.
			return new Short(currenttoken);
		}
		catch(NumberFormatException e) {
			// Return null because the token isn't a Short.
			return null;
		}
	}


	/**
	 * Returns whether there are more tokens to be read in this file.
	 * 
	 * @return Whether there are more tokens to be read in this file.
	 */
	public boolean fileHasMoreTokens() {
		// Check if there are more tokens on the current line...
		if(lineHasMoreTokens()) {
			// ...and indicate that there are more tokens in the file if so.
			return true;
		}
		// If there are no more tokens on this line...
		else {
			// ...return if there is a next non-empty line.
			if(nextline != null)  return true;
			else  return false;
		}
	}

	/**
	 * Returns whether there are more tokens to be read on this line.
	 * 
	 * @return Whether there are more tokens to be read on this line.
	 */
	public boolean lineHasMoreTokens() {
		// Return if there are more tokens on the current line.
		if(tokenizer != null)  return tokenizer.hasMoreTokens();
		// No tokenizer, end of file has been reached.
		else return false;
	}


	/**
	 * Returns whether the current token is the first one on the line.
	 * 
	 * @return Whether the current token is the first one on the line.
	 */
	public boolean firstToken() {
		// Construct a StringTokenizer over the current line.
		StringTokenizer st = new StringTokenizer(currentline, delimiters, returndelims);

		// Compare the amount of tokens left to read with the amount of tokens left to read on tokenizer.
		// If tokenizer is one smaller than st, the current token is the first.
		if(tokenizer != null)  return st.countTokens() == tokenizer.countTokens() + 1;
		// No tokenizer, end of file has been reached.
		else  return false;
	}

	/**
	 * Returns whether the current token is the last one on the line.
	 * 
	 * @return Whether the current token is the last one on the line.
	 */
	public boolean lastToken() {
		// Return the counter value of there being more elements on the current line.
		if(tokenizer != null)  return !tokenizer.hasMoreElements();
		// No tokenizer, end of file has been reached.
		else  return false;
	}


	/**
	 * The current token.
	 */
	protected String currenttoken;

	/**
	 * The current line.
	 */
	protected String currentline;

	/**
	 * The next line in the file.
	 */
	protected String nextline;

	/**
	 * The tokenizer delimiters.
	 */
	protected String delimiters;

	/**
	 * The return delimiters flag.
	 */
	protected boolean returndelims;

	/**
	 * The <code>BufferedReader</code> that wraps the file.
	 */
	protected BufferedReader reader;

	/**
	 * The <code>StringTokenizer</code> that is used to tokenize the current line.
	 */
	protected StringTokenizer tokenizer;
}