/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// This class' package. Duh ;^)
package kirk.io;

// Import the IO classes.
import java.io.*;

// Import the everruling ArrayList class.
import java.util.ArrayList;

// Import the Wildcard class.
import kirk.util.Wildcard;

/**
 * <p>This class reads a file line by line into an <code>Arraylist</code>.
 * Various methods are supplied to retrieve the lines.
 * The power of this class lies in the fact that text files can be randomly accessed,
 * in a line by line fashion.
 * Take note that the entire file is read upon construction. This also means no files
 * will be held while this class is active, and no closing is required. This has as
 * side-effect that the file can be changed by external means while this class is
 * active.</p>
 * 
 * <p>Lines can also be changed, added, or removed, after which the
 * <code>flush(String filename)</code> method will write all changes to a file. In
 * order to be able to do this, the
 * <code>FileLineList(String filename, boolean writable)</code> constructor must be
 * called with <code>writable == true</code> upon object construction.</p>
 * 
 * <p>Seeing that this class allows the file it originally read from to be changed
 * while this class is active, any changes made to that file during that time will be
 * lost when the list of lines is flushed to that file. However, you can, if you
 * wish, flush the list to another file instead. You always need to supply a filename
 * when calling the flush method, though.</p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 1.00
 */
public class FileLineList {
	/**
	 * Constructs a new <code>FileLineList</code> that reads from
	 * <code>filename</code>. Whether or not <code>FileLineList</code> can write is
	 * indicated by <code>writable</code>. 
	 * 
	 * @param filename The file to be read from.
	 * @param writable true if this <code>FileLineList</code> must allow editing
	 * methods, false otherwise.
	 */
	public FileLineList(String filename, boolean writable) throws IOException {
		// Set whether this FileLineList supports writing.
		setWritable(writable);
		
		// Now we read the file into the list line by line.
		// Construct a new BufferedReader around our file. We need to use FileReader
		// of course.
		BufferedReader reader = new BufferedReader(new FileReader(filename));
		
		// The line continually read from the file. Initialise it to the first line.
		String line = reader.readLine();
		
		// Go through the file while we are not at the end of the file (the read line
		// will be null then).
		while(line != null) {
			// Add this line to the list. We can't use the add method of this class
			// because it can only be used when writing is supported.
			linelist.add(line);
			
			// Read the next line.
			line = reader.readLine();
		}
		
		// We are done. Close the file.
		reader.close();
	}

	/**
	 * Constructs a new <code>FileLineList</code> that reads from
	 * <code>filename</code>. This <code>FileLineList</code> can only be used to read
	 * lines from the file.
	 * 
	 * @param filename The file to be read from.
	 */
	public FileLineList(String filename)  throws IOException {
		// Overload the first constructor, with filename and false to indicate
		// that writing isn't allowed. 
		this(filename, false);
	}


	/**
	 * Retrieves the number of lines that were in the file.
	 * 
	 * @return The number of lines that were in the file.
	 */
	public int getNumberOfLines() {
		// Return the size of the arraylist, since that's the number of lines!
		return linelist.size();
	}
	

	/**
	 * Indicates whether the list contains the passed line.
	 * 
	 * @param line The line to be searched for.
	 * @return <code>true</code> if the line was found, <code>false</code> otherwise.
	 */
	public boolean contains(String line) {
		// Return whether the list contains this line.
		return linelist.contains(line);
	}

	/**
	 * Checks whether the list contains the passed line, and returns the index of
	 * the first occurence of that line. 
	 * 
	 * @param line The line to be searched for.
	 * @return The index of the line, or -1 if it was not found.
	 */
	public int firstIndexOf(String line) {
		// Return the first occurence of line.
		return linelist.indexOf(line);
	}

	/**
	 * Checks whether the list contains the passed line, and returns the index of
	 * the last occurence of that line. 
	 * 
	 * @param line The line to be searched for.
	 * @return The index of the line, or -1 if it was not found.
	 */
	public int lastIndexOf(String line) {
		// Return the first occurence of line.
		return linelist.lastIndexOf(line);
	}
	
	
	/**
	 * <p>Returns a list (in the form of an <code>ArrayList</code>) with all the
	 * lines that comply with the passed pattern.
	 * <b>Note:</b> Any changes made to the returned list will not be reflected in
	 * this object.<p>
	 * 
	 * <p><ul>
	 * <li>* can be used to indicate any amount of characters, and</li>
	 * <li>? can be used to indicate a single character.</li>
	 * </ul></p> 
	 * 
	 * @param wildcard The <code>String</code> containing wildcards that make out the
	 * selection pattern.
	 * @return An </code>ArrayList</code> with the results. All the results are
	 * <code>Strings</code>.
	 */
	public ArrayList getList(String wildcard) {
		// Construct the list to be returned.
		ArrayList list = new ArrayList();
		
		// Go through the linelist of this object.
		for(int index = 0; index < getNumberOfLines(); index++) {
			// Retrieve the line.
			String line = get(index);
			
			// Create a new WildCard, with wildcard as its wildcard string.
			Wildcard selector = new Wildcard(wildcard);
			
			// Check whether it complies with the wildcard String, and if it does,
			// add it to the list.
			if(selector.test(line))  list.add(line);
		}
		
		// Return the list.
		return list;
	}

	/**
	 * Converts this <code>FileLineList</code> into an <code>Array</code> and returns
	 * that.
	 * 
	 * @return The <code>Array</code> of <code>String</code>s that consists of all
	 * the lines that were in the file.  
	 */
	public String[] toArray() {
		// Make a new array of Strings, with the size of the list.
		String[] lines = new String[linelist.size()];
		
		// Put all the elements of linelist into this array. ArrayList will cast them
		// for us! w00t!
		// And then, of course, return it. After we cast it... I hope this works.
		return (String[]) linelist.toArray(lines);
	}
	
	/**
	 * Retrieves the line at the specified index.
	 * 
	 * @param index The index of the line to be returned. 
	 * @return The String at <code>index</code>.
	 */
	public String get(int index) {
		// Return the line they want. That would the one on index, yo.
		// Don't forget to cast it to a String, since ArrayList only knows Objects. 
		return (String) linelist.get(index);
	}


	/**
	 * Add the passed line to the list.
	 * 
	 * @param line The line to be added.
	 * @throws WriteNotAllowedException If writing is not allowed by this instance.
	 */
	public void add(String line) throws WriteNotAllowedException {
		// Check whether this operation is allowed, and add the line if it is.
		if(isWritable())  linelist.add(line);
		// If it is not allowed, throw an exception.
		else  throw new WriteNotAllowedException();
	}

	/**
	 * Insert the passed line into the list at the passed index.
	 * All lines after and including <code>index</code> move down one.
	 * The new line will be at <code>index</code>.
	 * 
	 * @param line The line to be inserted.
	 * @param index The index of the line that is after the the inserted line after
	 * insertion.
	 * @throws WriteNotAllowedException If writing is not allowed by this instance.
	 */
	public void insert(String line, int index) throws WriteNotAllowedException {
		// Check whether this operation is allowed, and insert the line if it is.
		if(isWritable())  linelist.add(index, line);
		// If it is not allowed, throw an exception.
		else  throw new WriteNotAllowedException();
	}


	/**
	 * Replaces the line at <code>index</code> with <code>line</code>.
	 * Note that these lines shouldn't end with newline characters or anything.
	 * This class takes care of that.
	 * 
	 * @param index The index of the line to be replaced.
	 * @param line The line that will replace the current line. 
	 * @throws WriteNotAllowedException If writing is not allowed by this instance.
	 */
	public void replace(int index, String line) throws WriteNotAllowedException {
		// Check whether this operation is allowed, and replace the line if it is.
		if(isWritable())  linelist.set(index, line);
		// If it is not allowed, throw an exception.
		else  throw new WriteNotAllowedException();
	}

	/**
	 * Appends the line at <code>index</code> with <code>line</code>.
	 * Note that these lines shouldn't end with newline characters or anything.
	 * This class takes care of that.
	 * 
	 * @param index The index of the line to be appended.
	 * @param line The line that will append the current line. 
	 * @throws WriteNotAllowedException If writing is not allowed by this instance.
	 */
	public void append(int index, String line) throws WriteNotAllowedException {
		// Check whether this operation is allowed, and append the line if it is.
		// This is done by replacing the line with a new line constructed from the
		// current line and the passed line. Confusing, isn't it :^P
		if(isWritable())  replace(index, ((String) linelist.get(index)) + line);
		// If it is not allowed, throw an exception.
		else  throw new WriteNotAllowedException();
	}

	/**
	 * Removes the line at <code>index</code>.
	 * 
	 * @param index The index of the line to be removed.
	 * @throws WriteNotAllowedException If writing is not allowed by this instance.
	 */
	public void remove(int index) throws WriteNotAllowedException {
		// Check whether this operation is allowed, and remove the line if it is.
		if(isWritable())  linelist.remove(index);
		// If it is not allowed, throw an exception.
		else  throw new WriteNotAllowedException();
	}


	/**
	 * Writes the lines in this list to the passed file, if allowed. 
	 * 
	 * @param filename The file to be written to.
	 */
	public void flush(String filename) throws IOException, WriteNotAllowedException {
		// Check whether this operation is allowed.
		if(isWritable()) {
			// Construct a BufferedWriter around filename, using FileWriter as a medium.
			BufferedWriter writer = new BufferedWriter(new FileWriter(filename));
	
			// Go through linelist.
			for(int index = 0; index < getNumberOfLines() - 1; index++) {
				// Retrieve the current line and write it to the file.
				writer.write(get(index));
				
				// Write a newline to the file.
				writer.newLine();
			}
			
			// Retrieve the last line and write it to the file. This is because we don't
			// need to write a newline after this. 
			writer.write(get(getNumberOfLines() - 1));
			
			// Close the file.
			writer.close();
		}
		// If it is not allowed, throw an exception.
		else  throw new WriteNotAllowedException();
	}


	/**
	 * Retrieves whether this <code>FileLineList</code> supports writing or not.
	 * 
	 * @return boolean Whether this <code>FileLineList</code> supports writing or
	 * not.
	 */
	public boolean isWritable() {
		// Just frigging return it!
		return writesupported;
	}
	
	/**
	 * Sets whether this <code>FileLineList</code> supports writing or not.
	 * 
	 * @param writable Whether this <code>FileLineList</code> supports writing or
	 * not.
	 */
	protected void setWritable(boolean writable) {
		// Just frigging set it!
		writesupported = writable;
	}


	/**
	 * The List of lines that were in the file when it was read.
	 */
	protected ArrayList linelist = new ArrayList();
	
	/**
	 * Indicates whether this <code>FileLineList</code> supports writing.
	 */
	protected boolean writesupported;
}