/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// T3h package.
package kirk.gui.layout;

// Import GUI schuph.
import java.awt.*;

/**
 * <p>A custom <code>LayoutManager</code> that lays out <code>Component</code>s like
 * a vertical <code>FlowLayout</code> that does not do multiple columns. Each
 * <code>Component</code> is as wide as the <code>Container</code> minus insets minus
 * gaps, or as wide as it prefers to be if it's maximum value is smaller.</p>
 * 
 * <p>Classes that extend this class make out how high each <code>Component</code>
 * is, and where they are placed (vertically).</p>
 * 
 * <p>Therefore there are 3 abstract methods to override, which are also specified by
 * the <code>LayoutManager</code> interface. Also provided are methods to retrieve
 * information about the width.</p>
 * 
 * <p>Note that <code>Component</code>s are always on the left of the line. There is
 * also only one <code>Component</code> per line. If you want to align your
 * <code>Component</code>, or want more <code>Component</code>s on a line, put a
 * <code>Panel</code> on the line and go from there.</p>  
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 1.00
 * @see java.awt.FlowLayout
 */
public abstract class LineLayout implements LayoutManager {
	/**
	 * Constructs a new <code>LineLayout</code> with the given values for the gaps.
	 */
	public LineLayout(int horizontalgap, int verticalgap) {
		// Set the values for the gaps.
		setHorizontalGap(horizontalgap);
		setVerticalGap(verticalgap);
	}

	/**
	 * Constructs a new <code>LineLayout</code> with the given value for the vertical
	 * gap, and the default value for the horizontal gap.
	 */
	public LineLayout(int verticalgap) {
		// Overload the full constructor with the default value for the horizontal
		// gap, and the given value for the vertical gap. 
		this(5, verticalgap);
	}
		
	/**
	 * Constructs a new <code>LineLayout</code>. Use the default values for the gaps.
	 */
	public LineLayout() {
		// Overload the full constructor with default values for the gaps.
		this(5, 5);
	}
	
	
	/**
	 * Returns the horizontal gap.
	 * 
	 * @return The size of the horizontal gap.
	 */
	public final int getHorizontalGap() {
		// Return the size of the horizontal gap.
		return horizontalgap;
	}

	/**
	 * Sets the horizontal gap. The horizontal gap cannot be negative.
	 * 
	 * @param horizontalgap The size the horizontal gap has to be set to.
	 */
	public final void setHorizontalGap(int horizontalgap) {
		// Check whether the horizontal gap is zero or positive.
		if(horizontalgap >= 0) {
			// Set the size of the horizontal gap.
			this.horizontalgap = horizontalgap;
		}
		// Otherwise, throw an exception.
		else  throw new IllegalArgumentException("Cannot set horizontal gap: The given horizontal gap is negative.");

	}
	
	
	/**
	 * Returns the vertical gap.
	 * 
	 * @return The size of the vertical gap.
	 */
	public final int getVerticalGap() {
		// Return the size of the vertical gap.
		return verticalgap;
	}

	/**
	 * Sets the vertical gap. The vertical gap cannot be negative.
	 * 
	 * @param verticalgap The size the vertical gap has to be set to.
	 */
	public final void setVerticalGap(int verticalgap) {
		// Check whether the vertical gap is zero or positive.
		if(verticalgap >= 0) {
			// Set the size of the vertical gap.
			this.verticalgap = verticalgap;
		}
		// Otherwise, throw an exception.
		else  throw new IllegalArgumentException("Cannot set vertical gap: The given vertical gap is negative.");
	}
	
		
	/**
	 * Called by a <code>Container</code> that has this <code>LayoutManager</code> as
	 * <code>LayoutManager</code>. It should add the given <code>Component</code> to
	 * this <code>LayoutManager</code>, however that is not neccesary. We use the
	 * same approach as <code>FlowLayout</code>.
	 * 
	 * @see java.awt.LayoutManager#addLayoutComponent(java.lang.String,
	 * java.awt.Component)
	 * @see java.awt.FlowLayout
	 */
	public final void addLayoutComponent(String name, Component comp) {
		// Do nothing.
		
		// That was easy!
	}

	/**
	 * Called by a <code>Container</code> that has this <code>LayoutManager</code> as
	 * <code>LayoutManager</code>. It should remove the given <code>Component</code>
	 * to this <code>LayoutManager</code>, however that is not neccesary. We use the
	 * same approach as <code>FlowLayout</code>.
	 * 
	 * @see java.awt.LayoutManager#removeLayoutComponent(java.awt.Component)
	 * @see java.awt.FlowLayout
	 */
	public final void removeLayoutComponent(Component comp) {
		// Do nothing.
		
		// That was easy!
	}
	
	
	/**
	 * Returns the preferred size of the <code>Container</code> if it get's layed out
	 * by this <code>LayoutManager</code>.
	 * 
	 * @param parent The <code>Container</code> that the preferred size is wanted of.
	 * @see java.awt.LayoutManager#preferredLayoutSize(java.awt.Container)
	 */
	public abstract Dimension preferredLayoutSize(Container parent);
	
	/**
	 * Returns the minimum size of the <code>Container</code> if it get's layed out
	 * by this <code>LayoutManager</code>.
	 * 
	 * @param parent The <code>Container</code> that the minimum size is wanted of.
	 * @see java.awt.LayoutManager#minimumLayoutSize(java.awt.Container)
	 */
	public abstract Dimension minimumLayoutSize(Container parent);
	
	
	/**
	 * Lays out the given <code>Container</code>.
	 * 
	 * @param parent The <code>Container</code> to be layed out.
	 * @see java.awt.LayoutManager#layoutContainer(java.awt.Container)
	 */
	public abstract void layoutContainer(Container parent);
	
	
	/**
	 * Returns the width the passed <code>Component</code> should have if it is layed
	 * out by this class.
	 * 
	 * @return The width <code>Component</code> should have.
	 */
	public final int getWidth(Container parent, Component component) {
		// The variables that make out horizontal positions:
		Insets insets = parent.getInsets();
		int left = insets.left + getHorizontalGap();
		int right = insets.right + getHorizontalGap();
		
		// If the maximum size allows it, return the width of the container (minus
		// some things). Otherwise, return the preferred width. 
		if(component.getMaximumSize().width >= parent.getWidth() - left - right)  return parent.getWidth() - left - right;
		else  return component.getPreferredSize().width;
	}

	/**
	 * Returns the preferred width of the passed parent when it would be layed out by
	 * this class.
	 * 
	 * @return The preferred width of the <code>parent</code>.
	 */
	public final int getPreferredWidth(Container parent) {
		// The width to be returned:
		int width = 0;
		
		// The preferred width is the preferred width of the component with the
		// largest one. Width, that is ;^)
		// Of course, we should not forget about the gaps! 
		// So we go through the list of components.
		for(int index = 0; index < parent.getComponentCount(); index++) {
			// Check whether the current component is visible.
			if(parent.getComponent(index).isVisible()) {
				// Make the preferred width the larger of the 2 (current
				// component and current width). 
				width = Math.max(parent.getComponent(index).getPreferredSize().width, width);
			}
		}

		// Add the extra thingies.
		Insets insets = parent.getInsets();
		width += getHorizontalGap() * 2 + insets.left + insets.right;
				
		// Return the final width.
		return width;
	}
	
	/**
	 * Returns the minimum width of the passed parent when it would be layed out by
	 * this class.
	 * 
	 * @return The minimum width of the <code>parent</code>.
	 */
	public final int getMinimumWidth(Container parent) {
		// The width to be returned:
		int width = 0;
		
		// The minimum width is the minimum width of the component with the
		// largest one. Width, that is ;^)
		// Of course, we should not forget about the gaps! 
		// So we go through the list of components.
		for(int index = 0; index < parent.getComponentCount(); index++) {
			// Check whether the current component is visible.
			if(parent.getComponent(index).isVisible()) {
				// Make the minimum width the larger of the 2 (current
				// component and current width). 
				width = Math.max(parent.getComponent(index).getMinimumSize().width, width);
			}
		}

		// Add the extra thingies.
		Insets insets = parent.getInsets();
		width += getHorizontalGap() * 2 + insets.left + insets.right;
				
		// Return the final width.
		return width;
	}
	
	
	/**
	 * The horizontal gap.
	 */
	private int horizontalgap;

	/**
	 * The vertical gap.
	 */
	private int verticalgap;
}