/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// The package of these claz.
package kirk.gui.layout;

/**
 * This class represents a constraints object for <code>GridSpanLayout</code>.
 * Various methods are available to set the correct settings for each associated
 * <code>Component</code>. For more information, see <code>GridSpanLayout</code>.
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 1.00
 * @see kirk.gui.layout.GridSpanLayout
 * @see java.awt.GridBagConstraints
 */
public class GridSpanConstraints implements Cloneable {
	/**
	 * Constructs a new <code>GridSpanConstraints</code> with the passed settings.
	 * 
	 * @param x The x coordinate.
	 * @param y The y coordinate.
	 * @param width The width coordinate.
	 * @param height The height coordinate.
	 * @param horizontalfill The relative horizontal fill.
	 * @param verticalfill The relative vertical fill.
	 * @param horizontalanchor The horizontal relative position.
	 * @param verticalanchor The vertical relative position.
	 */
	public GridSpanConstraints(int x, int y, int width, int height, double horizontalfill, double verticalfill, double horizontalanchor, double verticalanchor) {
		// Set the values as they have been passed to this method.
		setX(x);
		setY(y);
		setWidth(width);
		setHeight(height);
		setHorizontalFill(horizontalfill);
		setVerticalFill(verticalfill);
		setHorizontalAnchor(horizontalanchor);
		setVerticalAnchor(verticalanchor);
	}


	/**
	 * Constructs a new <code>GridSpanConstraints</code> with the given settings, and
	 * the default values for the rest. See the no parameter constructor for the
	 * default values.
	 * 
	 * @param x The x coordinate.
	 * @param y The y coordinate.
	 * @param width The width coordinate.
	 * @param height The height coordinate.
	 */
	public GridSpanConstraints(int x, int y, int width, int height) {
		// Overload the full parameter constructor with values as they have been
		// passed to this method, and the default values for the rest.
		this(x, y, width, height, -1.0, -1.0, -1.0, -1.0);
	}

	
	/**
	 * Constructs a new <code>GridSpanConstraints</code> with the default settings.
	 * These are:
	 * <ul>
	 *   <li>0 for the x coordinate.
	 *   <li>0 for the y coordinate.
	 *   <li>1 for the width.
	 *   <li>1 for the height.
	 *   <li>-1.0 for the horizontal fill (makes the <code>LayoutManager</code> use
	 *       the <code>Component</code>'s preferred width.
	 *   <li>-1.0 for the the vertical fill (makes the <code>LayoutManager</code> use
	 *       the <code>Component</code>'s preferred height.
	 *   <li>-1.0 for the horizontal anchor (makes the <code>LayoutManager</code> use
	 *       the <code>Component</code>'s value.
	 *   <li>-1.0 for the vertical anchor (makes the <code>LayoutManager</code> use
	 *       the <code>Component</code>'s value.
	 * </ul>
	 */
	public GridSpanConstraints() {
		// Overload the full parameter constructor with the default values.
		this(0, 0, 1, 1, -1.0, -1.0, -1.0, -1.0);
	}


	/**
	 * Sets the x coordinate of the associated <code>Component</code>.
	 * 
	 * @param x The x coordinate.
	 */
	public void setX(int x) {
		// Set the x coordinate. We don't have to check whether it is in or out of
		// bounds, since we can't. GridSpanLayout will do that for us.
		this.x = x;
	}

	/**
	 * Returns the x coordinate of the associated <code>Component</code>.
	 * 
	 * @return The x coordinate.
	 */
	public int getX() {
		// Return the x coordinate.
		return x;
	}


	/**
	 * Sets the y coordinate of the associated <code>Component</code>.
	 * 
	 * @param y The y coordinate.
	 */
	public void setY(int y) {
		// Set the y coordinate. We don't have to check whether it is in or out of
		// bounds, since we can't. GridSpanLayout will do that for us.
		this.y = y;
	}

	/**
	 * Returns the y coordinate of the associated <code>Component</code>.
	 * 
	 * @return The y coordinate.
	 */
	public int getY() {
		// Return the y coordinate.
		return y;
	}
	
	
	/**
	 * Sets the width of the associated <code>Component</code>.
	 * 
	 * @param width The width.
	 */
	public void setWidth(int width) {
		// Set the width. We don't have to check whether it is in or outof
		// bounds, since we can't. GridSpanLayout will do that for us.
		this.width = width;
	}

	/**
	 * Returns the width of the associated <code>Component</code>.
	 * 
	 * @return The width.
	 */
	public int getWidth() {
		// Return the width.
		return width;
	}
		

	/**
	 * Sets the height of the associated <code>Component</code>.
	 * 
	 * @param height The height.
	 */
	public void setHeight(int height) {
		// Set the height. We don't have to check whether it is in or outof
		// bounds, since we can't. GridSpanLayout will do that for us.
		this.height = height;
	}

	/**
	 * Returns the height of the associated <code>Component</code>.
	 * 
	 * @return The height.
	 */
	public int getHeight() {
		// Return the height.
		return height;
	}
	
	
	/**
	 * Sets the horizontal fill of the associated <code>Component</code>.
	 * 
	 * @param horizontalfill The horizontal fill.
	 */
	public void setHorizontalFill(double horizontalfill) {
		// Set the horizontal fill.
		this.horizontalfill = horizontalfill;
	}

	/**
	 * Returns the horizontal fill of the associated <code>Component</code>.
	 * 
	 * @return The horizontal fill.
	 */
	public double getHorizontalFill() {
		// Return the horizontal fill.
		return horizontalfill;
	}
	
	
	/**
	 * Sets the vertical fill of the associated <code>Component</code>.
	 * 
	 * @param verticalfill The vertical fill.
	 */
	public void setVerticalFill(double verticalfill) {
		// Set the vertical fill.
		this.verticalfill = verticalfill;
	}

	/**
	 * Returns the vertical fill of the associated <code>Component</code>.
	 * 
	 * @return The vertical fill.
	 */
	public double getVerticalFill() {
		// Return the vertical fill.
		return verticalfill;
	}
	
	
	/**
	 * Sets the horizontal anchor of the associated <code>Component</code>.
	 * 
	 * @param horizontalanchor The horizontal anchor.
	 */
	public void setHorizontalAnchor(double horizontalanchor) {
		// Set the horizontal anchor.
		this.horizontalanchor = horizontalanchor;
	}

	/**
	 * Returns the horizontal anchor of the associated <code>Component</code>.
	 * 
	 * @return The horizontal anchor.
	 */
	public double getHorizontalAnchor() {
		// Return the horizontal anchor.
		return horizontalanchor;
	}
	
	
	/**
	 * Sets the vertical anchor of the associated <code>Component</code>.
	 * 
	 * @param verticalanchor The vertical anchor.
	 */
	public void setVerticalAnchor(double verticalanchor) {
		// Set the vertical anchor.
		this.verticalanchor = verticalanchor;
	}

	/**
	 * Returns the vertical anchor of the associated <code>Component</code>.
	 * 
	 * @return The vertical anchor.
	 */
	public double getVerticalAnchor() {
		// Return the vertical anchor.
		return verticalanchor;
	}
	
	
	/**
	 * Returns a copy of this object. 
	 * 
	 * @return A copy of this object.
	 * @see java.lang.Object#clone()
	 */
	public Object clone() {
		// Catch errors.
		try {
			// Return a cloned object.
			return super.clone();
		}
		catch(CloneNotSupportedException e) {
			// This shouldn't happen, since we are Cloneable.
			throw new InternalError("Clone not supported! " + e.getMessage());
		}
	}
	
	
	/**
	 * The X coordinate of the associated <code>Component</code>, in cells.
	 */
	protected int x;

	/**
	 * The Y coordinate of the associated <code>Component</code>, in cells.
	 */
	protected int y;
	
	
	/**
	 * The width of the associated <code>Component</code>, in cells.
	 */
	protected int width;
	
	/**
	 * The height of the associated <code>Component</code>, in cells.
	 */
	protected int height;
	
	
	/**
	 * Whether the associated <code>Component</code> should be as high as the space
	 * it is in.
	 */
	protected double horizontalfill;
	
	/**
	 * Whether the associated <code>Component</code> should be as wide as the space
	 * it is in.
	 */
	protected double verticalfill;


	/**
	 * The relative position of the associated <code>Component</code> on the x-axis
	 * of the space it is in.
	 */
	protected double horizontalanchor;
	
	/**
	 * The relative position of the associated <code>Component</code> on the y-axis
	 * of the space it is in.
	 */
	protected double verticalanchor;
}