/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// The package of this class. No really!
package kirk.gui.layout;

// Import gui stuff.
import java.awt.*;

/**
 * <p><code>GridLineLayout</code> extends <code>LineLayout</code>, and each line is
 * vertically treated like with <code>GridLayout</code>: The size is proportional to
 * the maximum size of the <code>Container</code> using this
 * <code>LayoutManager</code>, depending on how many lines there are.</p>
 * 
 * <p>Use this class if you want to line up multiple <code>LineLayouts</code> in a
 * resizable <code>Window</code> where the <code>Component</code>s are relatively
 * placed. Or something.</p>
 * 
 * <p>Note that <code>Component</code>s will have their preferred height, but with a
 * maximum of the line height. They will also be vertically centered in their line.
 * </p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 1.00
 * @since 1.00
 * @see kirk.gui.layout.LineLayout
 * @see kirk.gui.layout.VariableLineLayout
 * @see kirk.gui.layout.FixedLineLayout
 * @see kirk.gui.layout.LargestLineLayout
 */
public class GridLineLayout extends LineLayout {
	/**
	 * Construct a new <code>GridLineLayout</code> with the specified gap values.
	 */
	public GridLineLayout(int horizontalgap, int verticalgap) {
		// Call the superconstructor.
		super(horizontalgap, verticalgap);
	}

	/**
	 * Construct a new <code>GridLineLayout</code> with the default horizontal gap
	 * value and the specified vertical gap value.
	 */
	public GridLineLayout(int verticalgap) {
		// Call the superconstructor.
		super(verticalgap);
	}

	/**
	 * Construct a new <code>GridLineLayout</code> with the default gap values.
	 */
	public GridLineLayout() {
		// Call the superconstructor.
		super();
	}	
	
	
	/**
	 * Returns the preferred size of the <code>Container</code> if it get's layed out
	 * by this <code>LayoutManager</code>.
	 * 
	 * @param parent The <code>Container</code> that the preferred size is wanted of.
	 * @see kirk.gui.layout.LineLayout#preferredLayoutSize(java.awt.Container)
	 * @see java.awt.LayoutManager#preferredLayoutSize(java.awt.Container)
	 */
	public Dimension preferredLayoutSize(Container parent) {
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// The insets of the container.
			Insets insets = parent.getInsets();
			
			// Set the preferred height of the container to the amount of components
			// times the largest preferred height. Plus gaps, and insets.
			// Retrieve the preferred width.
			// Return a new Dimension object with this data.
			return new Dimension(getPreferredWidth(parent), parent.getComponentCount() * getLargestPreferredHeight(parent) + 2 * getVerticalGap() + insets.top + insets.bottom);
		}
	}
	
	/**
	 * Returns the minimum size of the <code>Container</code> if it get's layed out
	 * by this <code>LayoutManager</code>.
	 * 
	 * @param parent The <code>Container</code> that the minimum size is wanted of.
	 * @see kirk.gui.layout.LineLayout#minimumLayoutSize(java.awt.Container)
	 * @see java.awt.LayoutManager#minimumLayoutSize(java.awt.Container)
	 */
	public Dimension minimumLayoutSize(Container parent) {
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// The insets of the container.
			Insets insets = parent.getInsets();
			
			// Set the minimum height of the container to the amount of components
			// times the largest minimum height. Plus gaps, and insets.
			// Retrieve the minimum width.
			// Return a new Dimension object with this data.
			return new Dimension(getMinimumWidth(parent), parent.getComponentCount() * getLargestMinimumHeight(parent) + 2 * getVerticalGap() + insets.top + insets.bottom);
		}
	}
	
	
	/**
	 * Lays out the given <code>Container<code>.
	 * 
	 * @param parent The <code>Container</code> to be layed out.
	 * @see kirk.gui.layout.LineLayout#layoutContainer(java.awt.Container)
	 * @see java.awt.LayoutManager#layoutContainer(java.awt.Container)
	 */
	public void layoutContainer(Container parent) {
		// Allright. This is it. No way back now. Here goes nothing...
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// The variables that make out positions:
			Insets insets = parent.getInsets();
			int placement = insets.top + getVerticalGap();
			int left = insets.left + getHorizontalGap();
			
			// Retrieve the height of the cells. It is the height of the container
			// minus the insets and the vertical gap, divided by the number of
			// components, minus the gap size.
			int cellheight = (parent.getHeight() - insets.top - insets.bottom - getVerticalGap()) / parent.getComponentCount() - getVerticalGap();
			 
			// We need to go through all the components in this container... so
			// that's what we will do.
			for(int index = 0; index < parent.getComponentCount(); index++) {
				// Retrieve the current component.
				Component component = parent.getComponent(index);
				
				// Check whether the current component is visible.
				if(component.isVisible()) {
					// Retrieve the width the current component should have.
					int width = getWidth(parent, component);
					
					// Make the component either as high as it prefers, or as high as
					// maximally allowed if it is too big.
					int height = Math.min(component.getPreferredSize().height, cellheight);
					
					// Center the component vertically (in its own little cell).
					int top = placement + (cellheight - height) / 2;
					
					// Place and size the component. 
					// That is, put it under the previous component (or at the top if
					// it is the first one).
					component.setBounds(left, top, width, height);
					
					// Increase top with the preset height and the gap for the next
					// component.
					placement += cellheight + getVerticalGap();
				}
			}
		}
	}
	
	
	/**
	 * Returns the largest preferred height among the visible <code>Component</code>s
	 * in <code>parent</code>.
	 * 
	 * @param parent The <code>Container</code> of which the visible
	 * <code>Component</code>s are checked.
	 * @return The largest preferred height among the visible <code>Components</code>
	 * in <code>parent</code>.
	 */
	public int getLargestPreferredHeight(Container parent) {
		// The preferred height is the amount of components in this container
		// times the preferred height of component with the largest preferred
		// height. Preferred height. Height.
		// The variable with the largest preferred height found.
		int largestheight = 0;
		
		// Find the largest minipreferredmum height.
		for(int index = 0; index < parent.getComponentCount(); index++) {
			// Check whether the current component is visible.
			if(parent.getComponent(index).isVisible()) {
				// Select the maximum of the current largest preffered height and
				// the component height. 
				largestheight = Math.max(largestheight, parent.getComponent(index).getPreferredSize().height);
			}
		}
		
		// Return the largest preferred height.
		return largestheight;
	}
	
	/**
	 * Returns the largest minimum height among the visible <code>Component</code>s
	 * in <code>parent</code>.
	 * 
	 * @param parent The <code>Container</code> of which the visible
	 * <code>Component</code>s are checked.
	 * @return The largest minimum height among the visible <code>Components</code>
	 * in <code>parent</code>.
	 */
	public int getLargestMinimumHeight(Container parent) {
		// The minimum height is the amount of components in this container
		// times the minimum height of component with the largest minimum
		// height. minimum height. Height.
		// The variable with the largest minimum height found.
		int largestheight = 0;
		
		// Find the largest minimum height.
		for(int index = 0; index < parent.getComponentCount(); index++) {
			// Check whether the current component is visible.
			if(parent.getComponent(index).isVisible()) {
				// Select the maximum of the current largest minimum height and
				// the component height. 
				largestheight = Math.max(largestheight, parent.getComponent(index).getMinimumSize().height);
			}
		}
		
		// Return the largest minimum height.
		return largestheight;
	}
}