/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// The package of this class. No really!
package kirk.gui.layout;

// Import gui stuff.
import java.awt.*;

/**
 * <p><code>FixedLineLayout</code> extends <code>LineLayout</code>, and it makes each
 * <p>line is as high as specified when constructing an object of this class.</p>
 * 
 * <p>Use this class if you want to line up multiple <code>LineLayouts</code>.</p>
 * 
 * <p>Note that <code>Components</code> will have their preferred height, but with a
 * maximum of the preset height. They will also be vertically centered in their line.
 * </p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 1.00
 * @since 1.00
 * @see kirk.gui.layout.LineLayout
 * @see kirk.gui.layout.VariableLineLayout
 * @see kirk.gui.layout.GridLineLayout
 * @see kirk.gui.layout.LargestLineLayout
 */
public class FixedLineLayout extends LineLayout {
	/**
	 * Construct a new <code>FixedLineLayout</code> with the specified height, and
	 * the specified gap values.
	 */
	public FixedLineLayout(int height, int horizontalgap, int verticalgap) {
		// Call the superconstructor.
		super(horizontalgap, verticalgap);
		
		// Set the height value.
		setHeight(height);
	}

	/**
	 * Construct a new <code>FixedLineLayout</code> with the specified height, and
	 * the specified vertical gap, and the default horizontal gap value.
	 */
	public FixedLineLayout(int height, int verticalgap) {
		// Call the superconstructor.
		super(verticalgap);
		
		// Set the height value.
		setHeight(height);
	}
	
	/**
	 * Construct a new <code>FixedLineLayout</code> with the specified height, and
	 * the default gap values.
	 */
	public FixedLineLayout(int height) {
		// Call the superconstructor.
		super();
		
		// Set the height value.
		setHeight(height);
	}
	
	
	/**
	 * Returns the height value this <code>LayoutManager</code> uses.
	 * 
	 * @return The height value.
	 */
	public int getHeight() {
		return height;
	}
	
	/**
	 * Sets the height value this <code>LayoutManager</code> uses.
	 * 
	 * @param height The height of each <code>Component</code>.
	 */
	public void setHeight(int height) {
		this.height = height;
	}
	
	
	/**
	 * Returns the preferred size of the <code>Container</code> if it get's layed out
	 * by this <code>LayoutManager</code>.
	 * 
	 * @param parent The <code>Container</code> that the preferred size is wanted of.
	 * @see kirk.gui.layout.LineLayout#preferredLayoutSize(java.awt.Container)
	 * @see java.awt.LayoutManager#preferredLayoutSize(java.awt.Container)
	 */
	public Dimension preferredLayoutSize(Container parent) {
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// The insets of the container.
			Insets insets = parent.getInsets();
			
			// The preferred height is the amount of components in this container
			// times the preset height. Plus gaps and insets of course.
			// Return a new Dimension object constructed from this data.
			return new Dimension(getPreferredWidth(parent), parent.getComponentCount() * getHeight() + 2 * getVerticalGap() + insets.top + insets.bottom);
		}
	}
	
	/**
	 * Returns the minimum size of the <code>Container</code> if it get's layed out
	 * by this <code>LayoutManager</code>.
	 * 
	 * @param parent The <code>Container</code> that the minimum size is wanted of.
	 * @see kirk.gui.layout.LineLayout#minimumLayoutSize(java.awt.Container)
	 * @see java.awt.LayoutManager#minimumLayoutSize(java.awt.Container)
	 */
	public Dimension minimumLayoutSize(Container parent) {
		// The minimum height is the same as the preferred height.
		// Retrieve the preferred height.
		Dimension dimension = preferredLayoutSize(parent);
		
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// Retrieve the minimum width.
			dimension.width = getMinimumWidth(parent);
		}
		
		// Return the Dimension object.
		return dimension;
	}
	
	
	/**
	 * Lays out the given <code>Container</code>.
	 * 
	 * @param parent The <code>Container</code> to be layed out.
	 * @see kirk.gui.layout.LineLayout#layoutContainer(java.awt.Container)
	 * @see java.awt.LayoutManager#layoutContainer(java.awt.Container)
	 */
	public void layoutContainer(Container parent) {
		// Allright. This is it. No way back now. Here goes nothing...
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// The variables that make out positions:
			Insets insets = parent.getInsets();
			int placement = insets.top + getVerticalGap();
			int left = insets.left + getHorizontalGap();
			
			// We need to go through all the components in this container... so
			// that's what we will do.
			for(int index = 0; index < parent.getComponentCount(); index++) {
				// Retrieve the current component.
				Component component = parent.getComponent(index);
				
				// Check whether the current component is visible.
				if(component.isVisible()) {
					// Retrieve the width the current component should have.
					int width = getWidth(parent, component);
					
					// Make the component either as high as it prefers, or as high as
					// maximally allowed if it is too big.
					int height = Math.min(component.getPreferredSize().height, getHeight());
					
					// Center the component vertically (in its own little cell).
					int top = placement + (getHeight() - height) / 2;
					
					// Place and size the component. 
					// That is, put it under the previous component (or at the top if
					// it is the first one).
					component.setBounds(left, top, width, height);
					
					// Increase placement with the preset height and the gap for the
					// next component.
					placement += getHeight() + getVerticalGap();
				}
			}
		}
	}
	
	
	/**
	 * The height of each <code>Component</code>.
	 */
	protected int height;
}