/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// T3h package.
package kirk.gui.layout;

// Import GUI schuph.
import java.awt.*;

/**
 * <p><code>CenteredLineLayout</code> extends <code>VariableLineLayout</code>, and it
 * makes each component as high and wide as its preferred size. It also centers all
 * components horizontally.</p>
 * 
 * <p>Use this class if you don't want to waste space or if you want easy adding of
 * components under each other. This class is particularly useful for
 * <code>PopupDialogs</code>. Actually, it was written for that purpose.</p>
 * 
 * <p>This class is not a real <code>LineLayout</code>, since it manages the width of
 * its components itself... but it uses all the other inherited stuff. It even has
 * the same preferred and minimum size methods as <code>VariableLineLayout</code>.
 * </p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 1.10
 * @since 1.10
 * @see kirk.gui.layout.LineLayout
 * @see kirk.gui.layout.VariableLineLayout
 */
public class CenteredLineLayout extends VariableLineLayout {
	/**
	 * Constructs a new <code>CenteredLineLayout</code> with the given values for the
	 * gaps.
	 */
	public CenteredLineLayout(int horizontalgap, int verticalgap) {
		// Call the superconstructor.
		super(horizontalgap, verticalgap);
	}

	/**
	 * Constructs a new <code>CenteredLineLayout</code> with the given value for the
	 * vertical gap, and the default value for the horizontal gap.
	 */
	public CenteredLineLayout(int verticalgap) {
		// Call the superconstructor.
		super(verticalgap);
	}
		
	/**
	 * Constructs a new <code>CenteredLineLayout</code>. Use the default values for
	 * the gaps.
	 */
	public CenteredLineLayout() {
		// Call the superconstructor.
		super();
	}


	/**
	 * Lays out the given <code>Container</code>.
	 * 
	 * @param parent The <code>Container</code> to be layed out.
	 * @see kirk.gui.layout.LineLayout#layoutContainer(java.awt.Container)
	 * @see java.awt.LayoutManager#layoutContainer(java.awt.Container)
	 */
	public void layoutContainer(Container parent) {
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// The variables that make out positions:
			Insets insets = parent.getInsets();
			int top = insets.top + getVerticalGap();
			int left = insets.left + getHorizontalGap();
			int right = insets.right + getHorizontalGap();
			
			// We need to go through all the components in this container... so
			// that's what we will do.
			for(int index = 0; index < parent.getComponentCount(); index++) {
				// Retrieve the current component.
				Component component = parent.getComponent(index);
				
				// Check whether the current component is visible.
				if(component.isVisible()) {
					// Make the Component as wide as it prefers.
					int width = component.getPreferredSize().width;

					// Make the Component as high as it prefers.
					int height = component.getPreferredSize().height;
					
					// Center the Component horizontally.
					int placement = left + (parent.getWidth() - left - right - width) / 2;
					
					// Place and size the component. 
					// That is, put it under the previous component (or at the top if
					// it is the first one).
					component.setBounds(placement, top, width, height);
					
					// Increase top for the next component.
					top += height + getVerticalGap();
				}
			}
		}
	}
}