/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// T3h Package.
package kirk.gui.dialog;

// Import t3h AWT schtuff.
import java.awt.*;

// Import the LayoutManager this class uses.
import kirk.gui.layout.CenteredLineLayout;

/**
 * <p>An entension of <code>InputDialog</code> that asks the user for a single
 * <code>String</code>.</p>
 * 
 * <p>Use <code>SingleInputRetriever</code> to easily get a value from this object.
 * </p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 1.10
 * @see kirk.gui.dialog.Retriever
 * @see kirk.gui.dialog.SingleInputRetriever
 */
public class SingleInputDialog extends InputDialog {
	/**
	 * Constructs a new <code>SingleInputDialog<code> with the passed message.
	 * 
	 * @param owner The pwning <code>Frame</code>.
	 * @param retriever The <code>Retriever</code> that will retrieve the value.
	 * @param message The message to be displayed.
	 * @param columns The width of the input field in characters.
	 * @param amount The amount of characters the message should be broken off at, or
	 * 0 if no breaking should occur.
	 */
	public SingleInputDialog(Frame owner, Retriever retriever, String message, int columns, int amount) {
		// Call the super constructor.
		super(owner, retriever);
		
		// Set the layout to CenteredLineLayout.
		setLayout(new CenteredLineLayout());
		
		// Check whether breaking should occur.
		if(amount > 0) {
			// Convert the message to an array of lines.
			String[] newmessage = convertMessage(message, amount);
							
			// Add a new Panel to put the labels on.
			Panel messagepanel = new Panel(new CenteredLineLayout(0, 0));
			add(messagepanel);
			
			// Add all the labels to the panel.
			for(int index = 0; index < newmessage.length; index++) {
				// Put this part of the message on a label and add it to the panel.
				Label messagelabel = new Label(newmessage[index], Label.CENTER);
				messagepanel.add(messagelabel);
			}
		}
		else {
			// Add a label with the message.
			Label messagelabel = new Label(message, Label.CENTER);
			add(messagelabel);
		}
		
		// Add a textfield to this Dialog.
		inputfield = new TextField(columns);
		add(inputfield);
		
		// Make a Panel with an OK and a Cancel button on it.
		Panel buttonpanel = new Panel();
		
		// Add an OK button to this Panel.
		Button okbutton = new Button("OK");
		okbutton.setActionCommand("OK");
		okbutton.addActionListener(this);
		buttonpanel.add(okbutton);
		
		// Add a Cancel button to this Panel.
		Button cancelbutton = new Button("Cancel");
		cancelbutton.setActionCommand("CANCEL");
		cancelbutton.addActionListener(this);
		buttonpanel.add(cancelbutton);
		
		// Add the Panel.
		add(buttonpanel);
	}
	
	/**
	 * Constructs a new <code>SingleInputDialog<code> with the passed message.
	 * 
	 * @param owner The pwning <code>Frame</code>.
	 * @param retriever The <code>Retriever</code> that will retrieve the value.
	 * @param message The message to be displayed.
	 * @param columns The width of the input field in characters.
	 */
	public SingleInputDialog(Frame owner, Retriever retriever, String message, int columns) {
		// Overload to the first constructor with a 0 value for the amount so that it
		// ignores it.
		this(owner, retriever, message, columns, 0);
	}
	
	
	/**
	 * Returns the value of this <code>InputDialog</code>. In this case, a
	 * <code>String<code>.
	 * 
	 * @return The <code>String<code> the user entered in this
	 * <code>InputDialog</code>.
	 */
	public String getValue() {
		// Return the contents of the TextField.
		return inputfield.getText();
	}
	
	
	/**
	 * The input field.
	 */
	protected TextField inputfield;
}