/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// The package of this claz.
package kirk.gui.dialog;

// Import the GUI and event stuff.
import java.awt.*;
import java.awt.event.*;

// Import teh classes.
import java.util.ArrayList;
import java.util.StringTokenizer;

// Import the window closer.
import kirk.gui.closer.WindowCloser;

/**
 * <p>This class is an abstract class that other <code>Dialog</code>s can extend to
 * save on standard behaviour, like closing the <code>Window</code>. Also provided is
 * action listening for a <code>Button</code> with the "CLOSE" action command, so if
 * you add a <code>Button</code>, set "CLOSE" as its action command and set this
 * class as its <code>ActionListener</code>, then that <code>Button</code> will close
 * the <code>PopupDialog</code>.</p>
 * 
 * <p>The idea is that subclasses define the look of the popup. This is why there are
 * no settings done in this class about how the contents are layout out.</p>
 * 
 * <p>The constructor initialises some things for you:
 * <ul>
 * <li>This class has no <code>LayoutManager</code> set</li>
 * <li>It cannot be resized by the user</li>
 * <li>It is modal</li>
 * <li>It will appear in the middle of the screen</li>
 * <li>It will be closed by a <code>WindowCloser</code></li>
 * </ul></p>
 * 
 * <p>It also implements <code>ActionListener</code>, and it already implements
 * <code>actionPerformed</code> as well. However you can override it if you wish,
 * and/or call it by using <code>super()</code> for the standard "CLOSE" button
 * usage.</p>
 * 
 * <p>You can also close the dialog by calling <code>close()</code>. Useful for
 * subclasses.</p>
 * 
 *
 * @author Berend Wouda
 * @version 2.10
 * @since 1.10
 */
public abstract class PopupDialog extends Dialog implements ActionListener {
	/**
	 * Constructs a new <code>PopupDialog</code>.
	 * 
	 * @param owner The owner of this <code>Dialog</code>.
	 * @param title The title of this <code>Dialog</code>.
	 */
	public PopupDialog(Frame owner, String title) {
		// Call the supercontructor, with owner, the passed title and true for
		// "modal" (blocks input to the owner frame while this Dialog is open).
		super(owner, title, true);
		
		// Set the layout manager to nothing.
		setLayout(null);
		
		// Set it so that the user cannot resize the window.
		setResizable(false);
		
		// This screen has to be closable too, you know. We don't want it to end the
		// program, of course.
		addWindowListener(new WindowCloser());
	}


	/**
	 * Closes this <code>Dialog</code> if the action command equals "CLOSE".
	 * 
	 * @param event The standard passed <code>ActionEvent</code>.
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent event) {
		// Check if the action is "CLOSE".
		if(event.getActionCommand().equals("CLOSE")) {
			// This means this window can be closed.
			close();
		}
		// Any others can be taken care of by subclasses.
	}
	
	/**
	 * Closes this <code>Dialog</code>.
	 */
	protected void close() {
		// Hide this Dialog.
		hide();
			
		// Dump this Dialog.
		dispose();
	}
	
	
	/**
	 * Lays out this <code>Dialog</code>. The actual laying out is done by a
	 * <code>LayoutManager</code>, but this method first sets the correct size of the
	 * window by using the preferred sizes of its components.
	 * 
	 * @see java.awt.Component#doLayout()
	 */
	public void doLayout(){
		// Set the size of the window to the preferred size of this window according
		// to the LayoutManager.
		setSize(getPreferredSize());
		
		// Center the window.
		center();
		
		// Let the LayoutManager handle the rest.
		super.doLayout();
	}
	
	/**
	 * Centers the <code>Dialog</code> in the screen.
	 */
	protected void center() {
		// Set the location (in the middle of the screen).
		Dimension screensize = Toolkit.getDefaultToolkit().getScreenSize();
		setLocation((screensize.width - getWidth()) / 2, (screensize.height - getHeight())/ 2);
	}
	
	
	/**
	 * <p>Converts a message into a multiline format with <code>amount</code>
	 * characters on a line maximum, breaking lines on these delimiters: " \t\n\r\f".
	 * If there are undelimited tokens longer than <code>amount</code> characters
	 * they will occupy a single line.</p>
	 * 
	 * <p>This method is supplied for convenience, since many subclasses have use for
	 * it.</p>
	 * 
	 * @param message The message to convert.
	 * @param amount The maximum amount of characters on a line.
	 * @return The new message line by line in an array of Strings.
	 */
	protected String[] convertMessage(String message, int amount) {
		// Construct a new StringTokenizer, that also tokenizes delimiters.
		// Legacy class?? StringTokenizer rulez!
		StringTokenizer tokenizer = new StringTokenizer(message, " \t\n\r\f", true);
		
		// The multiline message.
		ArrayList lines = new ArrayList();
		
		// The current token.
		String token = null;

		// The current line.
		StringBuffer line = new StringBuffer();
		
		// Go through the tokens...
		while(tokenizer.hasMoreTokens()) {
			// Grab a token.
			token = tokenizer.nextToken();
			
			// Check the would-be length of the current line.
			if(line.length() + token.length() > amount) {
				// The line is full. Add it to the list.
				lines.add(line.toString());
				
				// Reset the line with the newest token.
				line = new StringBuffer(token);
			}
			else {
				// There is room. Add the token.
				line.append(token);
			}
		}
		
		// Add the rest to the list.
		lines.add(line.toString());
		
		// The entire message has been converted. Return the list as a String array.
		return (String[]) lines.toArray(new String[0]);
	}
}