/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// T3h Package.
package kirk.gui.dialog;

// Import t3h AWT.
import java.awt.*;
import java.awt.event.ActionEvent;

/**
 * <p>An extension of <code>PopupDialog</code> that asks the user for input. This
 * class is abstract, the actual way of input and the looks is decided by subclasses.
 * The method <code>actionPerformed</code> assumes there is an "OK" button, and/or a
 * "Cancel" button. If the action command specified by
 * <code>ActionEvent.getActionCommand()</code> is "OK", it calls the
 * <code>retrieve</code> method in the <code>Retriever</code> given on construction
 * time. This method should then retrieve any data wanted, and return. This class
 * will then close itself if <code>true</code> was returned by <code>retrieve</code>.
 * If the action command is "CANCEL", then the <code>PopupDialog</code> will be
 * closed without calling the <code>retrieve</code> method.</p>
 * 
 * <p>The implementation of <code>actionPerformed</code> in this class allows for
 * subclasses that only define a specific look for the input window, and that have
 * methods and/or attributes that can be used/retrieved by the <code>retrieve</code>
 * method. All buttons that they have should act like "OK" or "Cancel" buttons (see
 * above). If all these requirements have been met, the subclass doesn't have to
 * override <code>actionPerformed</code>. If it needs more buttons (say "Clear")
 * you need to override <code>actionPerformed</code>. You can then define your own
 * button usage, and use <code>super</code> to activate the "OK" and "Cancel" buttons
 * through this class.</p>
 * 
 * <p>If you close an <code>InputDialog</code> with the close button inherent to a
 * <code>Window</code>, closing is done through a standard <code>WindowCloser</code>.
 * <i>This means that this button has the same effect as a "Cancel" button</i>.
 * This means <code>retrieve</code> will <i>not</i> be called. This behaviour is by
 * design, always providing a means to cancel.</p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 1.10
 * @see kirk.gui.dialog.Retriever
 */
public abstract class InputDialog extends PopupDialog {
	/**
	 * Constructs a new <code>InputDialog</code> with the passed settings.
	 * 
	 * @param owner The pwning <code>Frame</code>.
	 * @param retriever The <code>Retriever</code> that retrieves the values from
	 * this object.
	 */
	public InputDialog(Frame owner, Retriever retriever) {
		// Call the super constructor.
		super(owner, "Input Dialog");
		
		// Set the retriever.
		this.retriever = retriever;
	}
	

	/**
	 * <p>This method gets called when an ActionEvent is generated. If the action
	 * command is "OK", it makes the <code>Retriever</code> retrieve the values it
	 * requires, and it then calls the <code>close()</code> method of
	 * <code>PopupDialog</code>, which closes this <code>Dialog</code>.</p>
	 * 
	 * <p>However, if the action command of the button is "CANCEL", the
	 * <code>Dialog</code> will be closed as if the close button was pressed. The
	 * <code>Retriever</code> will <i>not</i> be called.</p>
	 *  
	 * <p>Subclasses of this class should supercall this method when requiring this
	 * functionality.</p>
	 * 
	 * @param event The standard passed <code>ActionEvent</code>.
	 * @see kirk.gui.dialog.PopupDialog#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent event) {
		// Retrieve the action command.
		String command = event.getActionCommand();
		
		// Check if the action is "OK".
		if(command.equals("OK")) {
			// Make the retriever retrieve.
			// If true is returned, close this InputDialog.
			if(retriever.retrieve(this))  close();
		}
		// Check whether the action command is "CANCEL".
		else if(command.equals("CANCEL")) {
			// Close the window.
			close();
		}
		// Any others can be taken care of by subclasses.
	}
	
	
	/**
	 * The <code>Retriever</code>.
	 */
	protected Retriever retriever;
}