/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// The package. 
package kirk.gui.closer;

// Import the GUI and event stuff.
import java.awt.event.*;

// Import the GL4Java GLCanvas class.
import gl4java.awt.GLCanvas;

/**
 * <p>This class closes <code>Window</code>s that generate <code>WindowEvents</code>
 * asking for it. If specified, it can also end the program after the closing of a
 * <code>Window</code>. This is useful for main <code>Window</code>s.</p>
 * 
 * <p>This class is special, because it is specially made for <code>Window</code>s
 * having a <code>GLCanvas</code> or <code>GLAnimCanvas</code> on it. These need to
 * have <code>cvsDispose()</code> called before the <code>Window</code> closes.
 * These classes are from the GL4Java OpenGL binding.</p>
 * 
 * <p>For more control I recommend using the <code>Finisher</code> interface. That
 * way you can have key events close the program too (just have it call
 * <code>finish()</code>).</p>
 * 
 * @author Berend "Kirk" Wouda
 * @version 2.10
 * @since 2.10
 */
public class GLWindowCloser extends WindowCloser {
	/**
	 * Constructs a new <code>GLWindowCloser</code> with the exit option, and the
	 * given <code>GLCanvas</code>.
	 * 
	 * @param glcanvas The <code>GLCanvas</code> to dispose before the window it is
	 * in is disposed.
	 * @param exit <code>true</code> if the program should exit after closing a
	 * window, <code>false</code> otherwise.
	 */
	public GLWindowCloser(GLCanvas glcanvas, boolean exit) {
		// Call the super implementation.
		super(exit);
		
		// Set the canvas.
		this.glcanvas = glcanvas;
	}
	
	/**
	 * Constructs a new <code>GLWindowCloser</code> with the given
	 * <code>GLCanvas</code>. It will not end the program on close.
	 * 
	 * @param glcanvas The <code>GLCanvas</code> to dispose before the window it is
	 * in is disposed.
	 */
	public GLWindowCloser(GLCanvas glcanvas) {
		// Call the super implementation.
		super();
		
		// Set the canvas.
		this.glcanvas = glcanvas;
	}
	
	
	/**
	 * Closes the <code>Window</code>, and disposes the resources used by the
	 * <code>GLCanvas</code>.
	 * 
	 * @param event The event that indicates which <code>Window</code> to close.
	 */
	public void windowClosing(WindowEvent event) {
		// Dump the GLCanvas resources.
		glcanvas.cvsDispose();
		
		// Let the superclass do the rest.
		super.windowClosing(event);
	}
	
	
	/**
	 * The <code>GLCanvas</code> that needs a dispose before the window it is in is
	 * disposed. 
	 */
	protected GLCanvas glcanvas;
}