/*
 * Copyright 2003, 2004 Berend "Kirk" Wouda
 * 
 * This file is part of KirkPack.
 * 
 * KirkPack is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * KirkPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with KirkPack; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// t3h p4ck4g3.
package kirk.gui.closer;

// Import the GUI and event stuff.
import java.awt.event.WindowEvent;

/**
 * <p>This class closes <code>Window</code>s that generate <code>WindowEvents</code>
 * asking for it. Also, it calls <code>finish()</code> of the class specified on
 * construction (which must implement <code>Finisher</code>). This does not have to
 * be the same class as the <code>Window</code> that is being closed.
 * <code>finish()</code> can then do stuff before the window is being closed, and it
 * can even stop the window from being closed by returning <code>false</code>.</p>
 * 
 * <p>Note that this elaborate implementation does not do much more than saving you
 * of writing some <code>Window</code> closing code, but it adds to encapsulation and
 * you don't have to extend <code>WindowAdapter</code> if you want only closing
 * stuff. Thus you are able to extend another class. This saves a lot of unneccesary
 * code of implementing <code>WindowListener</code>. Also, it stays in line with just
 * adding a dedicated <code>WindowCloser</code>. This was all I could do to make me
 * feel I didn't write useless code (^^)</p>
 * 
 * @author Berend Wouda
 * @version 2.10
 * @since 2.00
 * @see java.awt.Window
 * @see kirk.gui.closer.Finisher
 */
public class FinishingCloser extends WindowCloser {
	/**
	 * Constructs a <code>FinishingCloser</code> with the passed
	 * <code>Finisher</code> and the given closing option.
	 * 
	 * @param exit <code>true</code> if the program should exit after closing a
	 * window, <code>false</code> otherwise.
	 * @param finisher The associated <code>Finisher</code>.
	 */
	public FinishingCloser(boolean exit, Finisher finisher) {
		// Call the superconstructor.
		super(exit);
		
		// Set closable as the object we will call the finish() method of.
		this.finisher = finisher;
	}
	
	/**
	 * Constructs a <code>FinishingCloser</code> with the passed
	 * <code>Finisher</code> that does not end the program after closing.
	 * 
	 * @param finisher The associated <code>Finisher</code>.
	 */
	public FinishingCloser(Finisher finisher) {
		// Call the superconstructor.
		super();
		
		// Set closable as the object we will call the finish() method of.
		this.finisher = finisher;
	}
	
	
	/**
	 * Makes the associated <code>Finisher</code> do its thing and closes the passed
	 * <code>Window</code>.
	 * 
	 * @param event The event that indicates which <code>Window</code> to close.
	 */
	public void windowClosing(WindowEvent event) {
		// Call the finish() method of the associated Finisher.
		// Only close the Window/end the program if finish() asks for it/allows
		// it.
		if(finisher.finish(event.getWindow()))  super.windowClosing(event);
	}
	
	
	/**
	 * The object we will call the finish() method of before closing the passed
	 * <code>Window</code>.
	 */
	protected Finisher finisher;
}