// T3h package.
package kirk.gui;

// Import GUI schuph.
import java.awt.*;

/**
 * VariableLineLayout extends LineLayout, and it makes each component as high as its
 * preferred size.
 * 
 * Use this class if you don't want to waste space or if you want easy adding of
 * components under each other (like in a Dialog or something).
 * 
 * @author Berend "Kirk" Wouda
 * @version 1
 * @see kirk.gui.LineLayout
 * @see kirk.gui.FixedLineLayout
 * @see kirk.gui.GridLineLayout
 * @see kirk.gui.LargestLineLayout
 */
public class VariableLineLayout extends LineLayout {
	/**
	 * Constructs a new VariableLineLayout with the given values for the gaps.
	 */
	public VariableLineLayout(int horizontalgap, int verticalgap) {
		// Call the superconstructor.
		super(horizontalgap, verticalgap);
	}

	/**
	 * Constructs a new VariableLineLayout with the given value for the vertical gap,
	 * and the default value for the horizontal gap.
	 */
	public VariableLineLayout(int verticalgap) {
		// Call the superconstructor.
		super(verticalgap);
	}
		
	/**
	 * Constructs a new VariableLineLayout. Use the default values for the gaps.
	 */
	public VariableLineLayout() {
		// Call the superconstructor.
		super();
	}
	
	
	/**
	 * Returns the preferred size of the container if it get's layed out by this
	 * LayoutManager.
	 * 
	 * @param parent The Container that the preferred size is wanted of.
	 * @see kirk.gui.LineLayout#preferredLayoutSize(java.awt.Container)
	 * @see java.awt.LayoutManager#preferredLayoutSize(java.awt.Container)
	 */
	public Dimension preferredLayoutSize(Container parent) {
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// Initialise the Dimensions we are going to return.
			// We will access its fields directly, because Sun apparently thought it
			// wise to make the methods return doubles... I don't really get the
			// Dimension2D object anyways...
			Dimension dimension = new Dimension(0, 0);
			
			// The preferred height is the sum of preferred heights of all component
			// in this container. Plus gaps of course.
			// So we go through the list of components.
			for(int index = 0; index < parent.getComponentCount(); index++) {
				// Check whether the current component is visible.
				if(parent.getComponent(index).isVisible()) {
					// Make the preferred height the current height plus the
					// preferred height of the current component, plus a gap.
					dimension.height += parent.getComponent(index).getPreferredSize().height + getVerticalGap();
				}
			}
			
			// Add all the other stuff.
			Insets insets = parent.getInsets();
			dimension.height += getVerticalGap() + insets.top + insets.bottom;
			dimension.width = getPreferredWidth(parent);
			 
			// Return the Dimension object.
			return dimension;
		}
	}
	
	/**
	 * Returns the minimum size of the container if it get's layed out by this
	 * LayoutManager.
	 * 
	 * @param parent The Container that the minimum size is wanted of.
	 * @see kirk.gui.LineLayout#minimumLayoutSize(java.awt.Container)
	 * @see java.awt.LayoutManager#minimumLayoutSize(java.awt.Container)
	 */
	public Dimension minimumLayoutSize(Container parent) {
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// Initialise the Dimension we are going to return.
			// We will access its fields directly, because Sun apparently thought it
			// wise to make the methods return doubles... I don't really get the
			// Dimension2D object anyways...
			Dimension dimension = new Dimension(0, 0);
			
			// The minimum height is the sum of minimum heights of all component in
			// this container. Plus gaps of course.
			// So we go through the list of components.
			for(int index = 0; index < parent.getComponentCount(); index++) {
				// Check whether the current component is visible.
				if(parent.getComponent(index).isVisible()) {
					// Make the minimum height the current height plus the minimum
					// height of the current component, plus a gap.
					dimension.height += parent.getComponent(index).getMinimumSize().height + getVerticalGap();
				}
			}
			
			// Add all the other stuff.
			Insets insets = parent.getInsets();
			dimension.height += getVerticalGap() + insets.top + insets.bottom;
			dimension.width = getMinimumWidth(parent);
			 
			// Return the Dimension object.
			return dimension;
		}
	}
	
	
	/**
	 * Lays out the given container.
	 * 
	 * @param parent The Container to be layed out.
	 * @see kirk.gui.LineLayout#layoutContainer(java.awt.Container)
	 * @see java.awt.LayoutManager#layoutContainer(java.awt.Container)
	 */
	public void layoutContainer(Container parent) {
		// Allright. This is it. No way back now. Here goes nothing...
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// The variables that make out positions:
			Insets insets = parent.getInsets();
			int top = insets.top + getVerticalGap();
			int left = insets.left + getHorizontalGap();
			
			// We need to go through all the components in this container... so
			// that's what we will do.
			for(int index = 0; index < parent.getComponentCount(); index++) {
				// Retrieve the current component.
				Component component = parent.getComponent(index);
				
				// Check whether the current component is visible.
				if(component.isVisible()) {
					// Retrieve the width the current component should have.
					int width = getWidth(parent, component);

					// Make the component as high as it prefers.
					int height = component.getPreferredSize().height;
					
					// Place and size the component. 
					// That is, put it under the previous component (or at the top if
					// it is the first one).
					component.setBounds(left, top, width, height);
					
					// Increase top for the next component.
					top += height + getVerticalGap();
				}
			}
		}
	}
}
