// Muhaha. The package.
package kirk.gui;

// Import AWT schtuff.
import java.awt.*;

// Import your daddy class.
import java.util.ArrayList; 

// Import HashMap. For the first time ever. I never used it before! :^o
import java.util.HashMap;

/**
 * This class is a custom <code>LayoutManager</code> that is kinda like a mix of
 * <code>GridLayout</code> and <code>GridBagLayout</code>. For me, it's not as
 * confusing as <code>GridBagLayout</code>, but it offers more functionality than
 * <code>GridLayout</code>.<br>
 * <br>
 * <code>GridSpanLayout</code> is contructed with width and height in cells, much
 * like <code>GridLayout</code>. The cells are proportional to the total width and
 * height of the container using this <code>LayoutManager</code>. Also, there are
 * horizontal and vertical gaps between components, that default to 5 pixels, but can
 * be specified during construction time or later on by calling the mutator methods
 * for them.<br>
 * <br>
 * Components are layed out on the grid much like with <code>GridBagLayout</code>:
 * they are allowed to span multiple cells. However, the difference is that
 * <code>GridSpanLayout</code> has a preset amount of cells, and that the sizes of
 * all the cells are the same. Also, there are no weights to worry about. Any
 * resizing is done exactly like <code>GridLayout</code> and
 * <code>GridLineLayout</code>: all cells are resized equivalently, and remain
 * relatively on the same position.<br>
 * <br>
 * This <code>LayoutManager</code> is one that uses constraints to lay out its
 * <code>Component</code>s, much like <code>GridBagLayout</code>. The constraints are
 * in the form of <code>GridSpanConstraints</code> objects, which contain the values
 * needed for laying out the component. Each component has an associated constraints
 * object.<br>
 * <br>
 * These are the settings you can give a <code>GridSpanConstraints</code> object.<br>
 * <br>
 * <ul>
 *   <li>setX(int) - Sets the horinzontal position of the left upper corner of the
 *       component, in cells.
 *   <li>setY(int) - Sets the vertical position of the left upper corner of the
 *       component, in cells.
 *   <li>setWidth(int) - Sets the width of the component, in cells.
 *   <li>setHeight(int) - Sets the height of the component, in cells.
 *   <li>setHorizontalFill(double) - Sets the relative width of the component. The
 *       number has to be between 0.0 and 1.0, where 0.0 indicates that the component
 *       has no width, 1.0 indicates to fully fill the available horizontal space,
 *       0.5 indicates that the component is as wide as halve the available space,
 *       and so on.
 *       If you specify a value that is out of the 0.0-1.0 bounds(inclusive), then
 *       the preferred width of the component (specified by
 *       <code>Component.getPreferredSize().width</code>) will be used.
 *   <li>setVerticalFill(double) - Sets the relative height of the component. The
 *       number has to be between 0.0 and 1.0, where 0.0 indicates that the component
 *       has no height, 1.0 indicates to fully fill the available vertical space,
 *       0.5 indicates that the component is as high as halve the available space,
 *       and so on.
 *       If you specify a value that is out of the 0.0-1.0 bounds(inclusive), then
 *       the preferred height of the component (specified by
 *       <code>Component.getPreferredSize().height</code>) will be used.
 *   <li>SetHorizontalAnchor(double) - Sets the relative position of the component on
 *       the x-axis. The number has to be between 0.0 and 1.0, where 0.0 (the
 *       default value) indicates that the component should be placed at the far most
 *       left side, 1.0 indicates that the component should be placed at the far most
 *       right side, 0.5 indicates that the component should be centered, and so on.
 *       If you specify a value that is out of the 0.0-1.0 bounds (inclusive), then
 *       the alignment value of the component (specified by
 *       <code>Component.getAlignmentX()</code>) will be used.
 *   <li>SetVerticalAnchor(double) - Sets the relative position of the component on
 *       the y-axis. The number has to be between 0.0 and 1.0, where 0.0 (the
 *       default value) indicates that the component should be placed at the far most
 *       upper side, 1.0 indicates that the component should be placed at the far
 *       most lower side, 0.5 indicates that the component should be centered, and so
 *       on. If you specify a value that is out of the 0.0-1.0 bounds (inclusive),
 *       then the alignment value of the component (specified by
 *       <code>Component.getAlignmentY()</code>) will be used.
 * </ul>
 * 
 * Like <code>GridBagLayout</code>, the <code>GridSpanConstraints</code> objects
 * are copied. This because when you add or set a constraint, this class checks
 * whether it is allowed. So you can't get to the inner
 * <code>GridSpanConstraints</code> objects because otherwise you could
 * (accidentally) break this <code>LayoutManager</code>.
 * <br>
 * This class does not check for conflicts among <code>GridSpanConstraints</code>,
 * simply because it can't (it has no knowledge of a parent <code>Container</code>
 * when setting constraints, it can be effectively used to lay out multiple
 * containers, as the interface prescribes), but for easeness sake there can be only
 * one <code>Component</code> per cell. So, if there are multiple Components that are
 * on the same cell, only the one that comes first in the list of the parent
 * <code>Container</code> (<code>Container.getComponent(int n)</code>) will be displayed.
 * So, take care when adding Components (just like you would with
 * <code>GridBagLayout</code>... I still don't know what that class' behaviour is in
 * this case. I still don't know/understand a lot about it actually... What <i>is</i>
 * it with those weights :^P).
 *  
 * @author Berend "Kirk" Wouda
 * @version 1
 * @see java.awt.GridBagLayout
 * @see java.awt.GridLayout
 * @see kirk.gui.LineLayout
 * @see kirk.gui.GridSpanConstraints
 */
public class GridSpanLayout implements LayoutManager2 {
	/**
	 * Creates a new GridSpanLayout with the given width, height, horizontal gap and
	 * vertical gap.
	 * 
	 * @param width The width of this grid in cells.
	 * @param height The height of this grid in cells.
	 * @param horizontalgap The horizontal gap between each cell and the cells and
	 * borders at the left and right of the screen.
	 * @param verticalgap The vertical gap between each cell and the cells and
	 * borders at the top and bottom of the screen.
	 */
	public GridSpanLayout(int width, int height, int horizontalgap, int verticalgap) {
		// Set the passed values.
		setWidth(width);
		setHeight(height);
		setHorizontalGap(horizontalgap);
		setVerticalGap(verticalgap);
		
		// Initialise the HashMap.
		componentmap = new HashMap();
	}

	/**
	 * Creates a new GridSpanLayout with the given width and height, and the default
	 * values (5) for the horizontal gap and vertical gap.
	 * 
	 * @param width The width of this grid in cells.
	 * @param height The height of this grid in cells.
	 */
	public GridSpanLayout(int width, int height) {
		// Overload with the passed values and the default values for the gaps.
		this(width, height, 5, 5);
	}
	
	
	/**
	 * Returns the width of this grid in cells.
	 * 
	 * @return The width of this grid in cells.
	 */
	public int getWidth() {
		// Return the width. Of this grid. In cells. As asked for. 
		return width;
	}
	
	/**
	 * Sets the width of this grid in cells. The width cannot be zero or negative.
	 * 
	 * @param width The width of this grid in cells.
	 */
	public void setWidth(int width) {
		// Check whether the width is positive.
		if(width > 0) {
			// Well, what are you waiting for??
			this.width = width;
		}
		// Otherwise, throw an exception.
		else  throw new IllegalArgumentException("Cannot set width: The given width is not positive.");
	}


	/**
	 * Returns the height of this grid in cells.
	 * 
	 * @return The height of this grid in cells.
	 */
	public int getHeight() {
		// Return the height. Of this grid. In cells. As asked for. 
		return height;
	}
	
	/**
	 * Sets the height of this grid in cells. The height cannot be zero or negative.
	 * 
	 * @param height The height of this grid in cells.
	 */
	public void setHeight(int height) {
		// Check whether the width is positive.
		if(height > 0) {
			// Well, what are you waiting for??
			this.height = height;
		}
		// Otherwise, throw an exception.
		else  throw new IllegalArgumentException("Cannot set height: The given height is not positive.");
	}
	

	/**
	 * Returns the horizontal gap.
	 * 
	 * @return The size of the horizontal gap.
	 */
	public int getHorizontalGap() {
		// Return the size of the horizontal gap.
		return horizontalgap;
	}

	/**
	 * Sets the horizontal gap. The horizontal gap cannot be negative.
	 * 
	 * @param The size the horizontal gap has to be set to.
	 */
	public void setHorizontalGap(int horizontalgap) {
		// Check whether the horizontal gap is zero or positive.
		if(horizontalgap >= 0) {
			// Set the size of the horizontal gap.
			this.horizontalgap = horizontalgap;
		}
		// Otherwise, throw an exception.
		else  throw new IllegalArgumentException("Cannot set horizontal gap: The given horizontal gap is negative.");
	}
	
	
	/**
	 * Returns the vertical gap.
	 * 
	 * @return The size of the vertical gap.
	 */
	public int getVerticalGap() {
		// Return the size of the vertical gap.
		return verticalgap;
	}

	/**
	 * Sets the vertical gap. The vertical gap cannot be negative.
	 * 
	 * @param The size the vertical gap has to be set to.
	 */
	public void setVerticalGap(int verticalgap) {
		// Check whether the vertical gap is zero or positive.
		if(verticalgap >= 0) {
			// Set the size of the vertical gap.
			this.verticalgap = verticalgap;
		}
		// Otherwise, throw an exception.
		else  throw new IllegalArgumentException("Cannot set vertical gap: The given vertical gap is negative.");
	}
	
	
	/**
	 * Check whether the passed constraints are out of bounds.
	 * 
	 * @param constraints The constraints to be checked.
	 * @return <code>true</code> if the passed constraints are out of bounds,
	 * <code>false</code> otherwise.
	 */
	protected boolean outOfBounds(GridSpanConstraints constraints) {
		// Check the coordinates for out-of-bounds-ness.
		if(constraints.getX() < 0)  return true;
		if(constraints.getX() + constraints.getWidth() > getWidth())  return true;
		if(constraints.getY() < 0)  return true;
		if(constraints.getY() + constraints.getHeight() > getHeight())  return true;
		
		// If we get here everything is ok.
		return false;
	}
	
	
	/**
	 * Returns the <code>GridSpanConstraints</code> that is associated with the passed
	 * key. If no such object exists, a default one is associated with the passed key,
	 * and returned. This because it is possible that key are passed that do not exist
	 * in the mapping. This happens when a <code>Container</code> sets this object as
	 * its <code>LayoutManager</code> after adding <code>Component</code>s.
	 * <code>null</code> is <i>not</i> allowed as key.
	 * 
	 * @param component The key that is associated with the
	 * <code>GridSpanConstraints</code>.
	 * @return The <code>GridSpanConstraints</code> that are associated with the
	 * given key.
	 */
	public GridSpanConstraints getConstraints(Component component) {
		// Check whether the key is null, and if it is, throw an exception.
		if(component == null)  throw new IllegalArgumentException("Cannot retrieve the constraints object from the mapping: The given key is null.");
		// If it is not null we can proceed.
		else {
			// Retrieve the Constraints from the Hashmap.
			GridSpanConstraints constraints = (GridSpanConstraints) componentmap.get(component);
			
			// Check whether the retrieved Constraints is not null and return it.
			// Otherwise we associate the Component with a default Constraints and return
			// that. This because it is possible that keys are supplied that are not in
			// the map (for example in the case when this LayoutManager is set to a
			// Container which already has components.
			if(constraints != null)  return (GridSpanConstraints) constraints.clone();
			else {
				constraints = new GridSpanConstraints();
				setConstraints(component, constraints);
				return (GridSpanConstraints) constraints.clone();
			}
		} 
	}
	
	/**
	 * Adds a new association to the internal mapping. Note that if null is passed
	 * as the associated <code>GridSpanConstraints</code>, a default
	 * <code>GridSpanConstraints</code> will be returned when the passed
	 * <code>Component</code> is given as key
	 * <code>null</code> is <i>not</i> allowed as key.
	 * 
	 * @param component The key of the association.
	 * @param constraints The associated <code>GridSpanConstraints</code>.
	 * @return The previously associated <code>GridSpanConstraints</code>, if there
	 * was any.
	 */
	public GridSpanConstraints setConstraints(Component component, GridSpanConstraints constraints) {
		// Check whether the key is null, and if it is, throw an exception.
		if(component == null)  throw new IllegalArgumentException("Cannot add the constraints object to the mapping: The given key is null.");
		// If it is not null the contraints can be added.
		else {
			// Check whether the contraints is null, and add a default contraints if
			// it is.
			if(constraints == null) {
				// Put a default constraints in the map and return the previous
				// associated constraints.
				return (GridSpanConstraints) componentmap.put(component, new GridSpanConstraints());
			}
			// If it is not null than the given constraint can be added.
			else {
				// Check whether these constraints are out of bounds. If they are,
				// throw an exception.
				if(outOfBounds(constraints)) throw new IllegalArgumentException("Cannot add the constraints object to the mapping: The constraints are out of bounds.");
				// If they aren't, the contraints can be added.
				else {
					// Put the association in the map and return the previous associated
					// constraints.
					return (GridSpanConstraints) componentmap.put(component, constraints.clone());
				}
			}
		}
	}
	
	/**
	 * Removes the association indicated by the component from the internal mapping.
	 * <code>null</code> is <i>not</i> allowed as key.
	 * 
	 * @param component The key of the association.
	 * @return The <code>GridSpanConstraints</code> that was associated with the key.
	 */
	public GridSpanConstraints removeConstraints(Component component) {
		// Check whether the key is null, and if it is, throw an exception.
		if(component == null)  throw new IllegalArgumentException("Cannot remove the constraints object from the mapping: The given key is null.");
		// If it isn't null, we can proceed.
		else {
			// Remove the association from the map and return the associated Constraints.
			return (GridSpanConstraints) componentmap.remove(component);
		}
	}
	
	
	/**
	 * Adds the passed <code>Component</code> to the LayoutManager with the passed
	 * name.
	 * 
	 * @see java.awt.LayoutManager#addLayoutComponent(java.lang.String,
	 * java.awt.Component)
	 */
	public void addLayoutComponent(String name, Component comp) {
		// I'm not sure if this is required, but just in case someone tries to add a
		// Component with this method, I'll overload it to the correct method with a
		// null. That ultimately makes it use a default constraints object. 
		addLayoutComponent(comp, null);
	}

	/**
	 * Adds the passed <code>Component</code> to the <code>LayoutManager</code> with
	 * the passed constraints. These constraints <i>must</i> be an instance of
	 * <code>GridSpanConstraints</code>.
	 * 
	 * @param comp The <code>Component</code> to be added.
	 * @param constraints The constraints to be associated with the passed
	 * <code>Component</code>.
	 * @see java.awt.LayoutManager2#addLayoutComponent(java.awt.Component,
	 * java.lang.Object)
	 */
	public void addLayoutComponent(Component comp, Object constraints) {
		// This is where the real work happens. Well, it's not too much work...
		// We just add the Constraint to the internal mapping with the Component as
		// key.
		// Of course, we need to check first whether constraints is what we require.
		if(constraints instanceof GridSpanConstraints || constraints == null) {
			// Add the association. Cast constraints to the correct object.
			setConstraints(comp, (GridSpanConstraints) constraints);
		}
		// A wrong object was used as constraints. Throw an exception to indicate
		// this.
		else  throw new IllegalArgumentException("Cannot add Component to LayoutManager: The given constraints obejct was not an instance of GridSpanConstraints.");
	}
	
	/**
	 * Removes the passed <code>Component</code> from the LayoutManager.
	 * 
	 * @param comp The <code>Component</code> to be removed.
	 * @see java.awt.LayoutManager#removeLayoutComponent(java.awt.Component)
	 */
	public void removeLayoutComponent(Component comp) {
		// Remove the assocation from the internal mapping.
		removeConstraints(comp);
	}


	/**
	 * Returns the size of the passed <code>Container</code> as this
	 * <code>LayoutManager</code> would like it best when the passed
	 * <code>Container</code> where to be layed out with this
	 * <code>LayoutManager</code>. 
	 * 
	 * @param target The <code>Container</code> who is te be checked for preferred
	 * size.
	 * @return The preferred size of this <code>Container</code>.
	 * @see java.awt.LayoutManager#preferredLayoutSize(java.awt.Container)
	 */
	public Dimension preferredLayoutSize(Container parent) {
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// Make a grid. Use passive mode.
			Component[][] componentgrid = makeGrid(parent, false);
			
			// The array of cell widths.
			Integer[][] cellwidths = new Integer[getWidth()][getHeight()];

			// The array of cell heights.
			Integer[][] cellheights = new Integer[getWidth()][getHeight()];
			
			// Go through the rows.
			for(int row = 0; row < getHeight(); row++) {	
				// Go through the columns.
				for(int column = 0; column < getWidth(); column++) {
					// Retrieve the component at the current grid position.
					Component component = componentgrid[column][row];
					
					// Check whether the current column is the last column, or
					// whether the next column does not contain the same component
					// as this one.
					if(column == getWidth() - 1 || componentgrid[row][column + 1] != component) {
						// If so, retrieve the component at the current grid position
						// and put its width at the same place in the Array of cell
						// widths.
						cellwidths[column][row] = new Integer(component.getPreferredSize().width);
					}
					// If not so, then this is a cell with a component spanned on it,
					// and it is not the last one of the span. We put in null to
					// indicate this.
					else  cellwidths[column][row] = null;
				}
			}
			
			// Go through the columns.
			for(int column = 0; column < getWidth(); column++) {	
				// Go through the rows.
				for(int row = 0; row < getHeight(); row++) {
					// Retrieve the component at the current grid position.
					Component component = componentgrid[column][row];
					
					// Check whether the current row is the last row, or
					// whether the next row does not contain the same component
					// as this one.
					if(row == getHeight() - 1 || componentgrid[row + 1][column] != component) {
						// If so, retrieve the component at the current grid position
						// and put its height at the same place in the Array of cell
						// heights.
						cellheights[column][row] = new Integer(component.getPreferredSize().height);
					}
					// If not so, then this is a cell with a component spanned on it,
					// and it is not the last one of the span. We put in null to
					// indicate this.
					cellheights[column][row] = null;
				}
			}

			// The insets of the parent container.
			Insets insets = parent.getInsets();
			
			// Return the preferred size.
			// Add the extra padding thingies.
			return new Dimension(getGridWidth(cellwidths) + getWidth() * (getHorizontalGap() + 1) + insets.left + insets.right, getGridHeight(cellheights) + getHeight() * (getVerticalGap() + 1) + insets.top + insets.bottom);
		}
	}

	/**
	 * Returns the size of the passed <code>Container</code> as this
	 * <code>LayoutManager</code> would like it to be at minimum when the passed
	 * <code>Container</code> where to be layed out with this
	 * <code>LayoutManager</code>. 
	 * 
	 * @param target The <code>Container</code> who is te be checked for minimum
	 * size.
	 * @return The minimum size of this <code>Container</code>.
	 * @see java.awt.LayoutManager#minimumLayoutSize(java.awt.Container)
	 */
	public Dimension minimumLayoutSize(Container parent) {
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// Make a grid. Use passive mode.
			Component[][] componentgrid = makeGrid(parent, false);
			
			// The array of cell widths.
			Integer[][] cellwidths = new Integer[getWidth()][getHeight()];

			// The array of cell heights.
			Integer[][] cellheights = new Integer[getWidth()][getHeight()];
			
			// Go through the rows.
			for(int row = 0; row < getHeight(); row++) {	
				// Go through the columns.
				for(int column = 0; column < getWidth(); column++) {
					// Retrieve the component at the current grid position.
					Component component = componentgrid[column][row];
					
					// Check whether the current column is the last column, or
					// whether the next column does not contain the same component
					// as this one.
					if(column == getWidth() - 1 || componentgrid[row][column + 1] != component) {
						// If so, retrieve the component at the current grid position
						// and put its width at the same place in the Array of cell
						// widths.
						cellwidths[column][row] = new Integer(component.getMinimumSize().width);
					}
					// If not so, then this is a cell with a component spanned on it,
					// and it is not the last one of the span. We put in null to
					// indicate this.
					cellwidths[column][row] = null;
				}
			}
			
			// Go through the columns.
			for(int column = 0; column < getWidth(); column++) {	
				// Go through the rows.
				for(int row = 0; row < getHeight(); row++) {
					// Retrieve the component at the current grid position.
					Component component = componentgrid[column][row];
					
					// Check whether the current row is the last row, or
					// whether the next row does not contain the same component
					// as this one.
					if(row == getHeight() - 1 || componentgrid[row + 1][column] != component) {
						// If so, retrieve the component at the current grid position
						// and put its height at the same place in the Array of cell
						// heights.
						cellheights[column][row] = new Integer(component.getMinimumSize().height);
					}
					// If not so, then this is a cell with a component spanned on it,
					// and it is not the last one of the span. We put in null to
					// indicate this.
					cellheights[column][row] = null;
				}
			}

			// The insets of the parent container.
			Insets insets = parent.getInsets();
			
			// Return the minimum size.
			// Add the extra padding thingies.
			return new Dimension(getGridWidth(cellwidths) + getWidth() * (getHorizontalGap() + 1) + insets.left + insets.right, getGridHeight(cellheights) + getHeight() * (getVerticalGap() + 1) + insets.top + insets.bottom);
		}
	}

	/**
	 * Returns the size of the passed <code>Container</code> as this
	 * <code>LayoutManager</code> would like it to be at maximum when the passed
	 * <code>Container</code> where to be layed out with this
	 * <code>LayoutManager</code>. 
	 * 
	 * @param target The <code>Container</code> who is te be checked for maximum
	 * size.
	 * @return The maximum size of this <code>Container</code>.
	 * @see java.awt.LayoutManager2#maximumLayoutSize(java.awt.Container)
	 */
	public Dimension maximumLayoutSize(Container target) {
		// This LayoutManager can layout any Container within the Java bounds.
		return new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);
	}


	/**
	 * Returns the alignment along the x-axis of the passed container if it where to
	 * be layed out with this <code>LayoutManager</code>.
	 * Always returns that it wants to be centered.
	 * 
	 * @param target The <code>Container</code> who is te be checked for preference.
	 * @return 0.5, to indicate it wants to be centered. 
	 * @see java.awt.LayoutManager2#getLayoutAlignmentX(java.awt.Container)
	 */
	public float getLayoutAlignmentX(Container target) {
		// Return the float that indicates centering.
		return 0.5f;
	}

	/**
	 * Returns the alignment along the y-axis of the passed container if it where to
	 * be layed out with this <code>LayoutManager</code>.
	 * Always returns that it wants to be centered.
	 * 
	 * @param target The <code>Container</code> who is te be checked for preference.
	 * @return 0.5, to indicate it wants to be centered. 
	 * @see java.awt.LayoutManager2#getLayoutAlignmentY(java.awt.Container)
	 */
	public float getLayoutAlignmentY(Container target) {
		// Return the float that indicates centering.
		return 0.5f;
	}


	/**
	 * Invalidates the layout, indicating that if the layout manager
	 * has cached information it should be discarded.
	 * 
	 * @param target The <code>Container</code> that has to invalidated. 
	 * @see java.awt.LayoutManager2#invalidateLayout(java.awt.Container)
	 */
	public void invalidateLayout(Container target) {
		// We do not differentiate any containers, so their is no cached information.
	}


	/**
	 * Lays out the passed container with this LayoutManager, and the constraints
	 * objects associated with the Components of this container.
	 * 
	 * @param target The <code>Container</code> to be layed out.
	 * @see java.awt.LayoutManager#layoutContainer(java.awt.Container)
	 */
	public void layoutContainer(Container parent) {
		// Lock the thread monitor.
		synchronized(parent.getTreeLock()) {
			// I wonder if this will be easier or harder than the size methods.
			// First, we make a grid. Use active mode, we want the conflicting
			// Components gone.
			Component[][] componentgrid = makeGrid(parent, true);
			
			// Calculate the grid size in pixels.
			// We start off with one gap, to keep the grid size consistant. Later on,
			// the gap will be substracted from the cell size, and all the cells will
			// be the same size, and correctly positioned within the Container.
			Dimension parentsize = parent.getSize();
			Insets insets = parent.getInsets();
			int gridwidth = parentsize.width - insets.right - insets.left - getHorizontalGap();
			int gridheight = parentsize.height - insets.bottom - insets.top - getVerticalGap();
			
			// Calculate the starting positions.
			int top = insets.left + getHorizontalGap();
			int left = insets.top + getVerticalGap();
			
			// Calculate the cell size.
			int cellwidth = gridwidth / getWidth() - getHorizontalGap();
			int cellheight = gridheight / getHeight() - getVerticalGap();
			
			// The list of Components we have already placed.
			// You see, while going through the grid, we encounter the top left cell
			// of a Component always before other cells it spans. So, we can
			// immediately size and place it, and we add it to this list.
			// If a Component is already in this list, then it will not be placed
			// again!
			ArrayList componentlist = new ArrayList();
			
			// Now what?
			// Go through the grid. Rows first (or actually, last).
			for(int row = 0; row < getHeight(); row++) {	
				// Go through the columns.
				for(int column = 0; column < getWidth(); column++) {
					// Retrieve the component at the current grid position.
					Component component = componentgrid[column][row];
					
					// Check whether there is a Component at the current grid
					// position, and that the component is not already placed.
					if(component != null && !componentlist.contains(component)) {
						// Add the Component to the list. This indicates that this
						// Component has been placed. Well, will be now, actually.
						componentlist.add(component);
						
						// Retrieve the associated constraints object.
						GridSpanConstraints constraints = getConstraints(component);


						// The width of component:
						int width;
						
						// The width of the space assigned to component:
						int spannedwidth = (cellwidth + getHorizontalGap()) * constraints.getWidth() - getHorizontalGap();
						
						// The horizontal fill:
						double horizontalfill = constraints.getHorizontalFill();
							
						// Check whether the horizontal fill is out of bounds.
						if(horizontalfill < 0.0 || horizontalfill > 1.0) {
							// Make the width of the current Component its preferred
							// width, or the spanned width if it is bigger than the
							// spanned width.
							width = Math.min(component.getPreferredSize().width, spannedwidth);
						}
						else {
							// Set the width proportionate to the spanned width.
							width = (int) (horizontalfill * spannedwidth);
						}
						
						
						// The height of component:
						int height;
						
						// The height of the space assigned to component:
						int spannedheight = (cellheight + getVerticalGap()) * constraints.getHeight() - getVerticalGap();
						
						// The vertical fill:
						double verticalfill = constraints.getVerticalFill();
			
						// Check whether the vertical fill is out of bounds.
						if(verticalfill < 0.0 || verticalfill > 1.0) {
							// Make the height of the current Component its preferred
							// height, or the spanned height if it is bigger than the
							// spanned height.
							height = Math.min(component.getPreferredSize().height, spannedheight);
						}
						else {
							// Set the height proportionate to the spanned height.
							height = (int) (horizontalfill * spannedheight);
						}
						
						
						// The horizontal anchor:
						double horizontalanchor = constraints.getHorizontalAnchor();
							
						// Check whether the horizontal anchor is out of bounds.
						if(horizontalanchor < 0.0 || horizontalanchor > 1.0) {
							// Set the anchor to the preferred alignment of the
							// Component.
							// Automagically transforms from a float into a
							// double. Yay! :^P
							horizontalanchor = component.getAlignmentX();
						}
							
						// Calculate the relative horizontal position of the current
						// Component within its assigned space.
						double horizontalposition = horizontalanchor * (spannedwidth - width);
							
						// Set the horizontal position of the current Component
						// to the left edge of the current cell, plus the
						// position within the cell. 
						int x = left + column * (cellwidth + getHorizontalGap()) + (int) horizontalposition;
						
						
						// The vertical anchor:
						double verticalanchor = constraints.getVerticalAnchor();
							
						// Check whether the vertical anchor is out of bounds.
						if(verticalanchor < 0.0 || verticalanchor > 1.0) {
							// Set the anchor to the preferred alignment of the
							// Component.
							// Automagically transforms from a float into a
							// double. Yay! :^P
							verticalanchor = component.getAlignmentY();
						}
							
						// Calculate the relative vertical position of the current
						// Component within its assigned space.
						double verticalposition = verticalanchor * (spannedheight - height);
							
						// Set the vertical position of the current Component
						// to the upper edge of the current cell, plus the
						// position within the cell. 
						int y = top + row * (cellheight + getVerticalGap()) + (int) verticalposition;
						
						
						// Set the location and size of the current Component.
						component.setBounds(x, y, width, height);
					}
				}
			}
		}
	}
	
	
	/**
	 * Returns the grid. It is represented as an <code>Array</code> with
	 * <code>Component</code>s.
	 * 
	 * On each grid cell, there is one or no <code>Component</code>. If there
	 * are <code>Component</code>s that span multiple cells, they have multiple
	 * entries in the grid. If there are multiple <code>Component</code>s in one
	 * cell, only the one that appears first in the parent <code>Container</code>
	 * will appear. Note that only visible <code>Component</code>s will appear in the
	 * grid.
	 * 
	 * @param parent The <code>Container</code> that supplies the
	 * <code>Component</code> we make a grid of.
	 * @param active A boolean indicating whether this method should make unused
	 * <code>Component</code>s invisible. <code>layoutContainer()</code> calls this
	 * as true, the size methods call this as false.
	 * @return A two-dimensional <code>Array</code> of Components that represents the
	 * grid.
	 */
	protected Component[][] makeGrid(Container parent, boolean active) {
		// Construct a new Array with the width and height of this LayoutManager.
		Component[][] componentgrid = new Component[getWidth()][getHeight()];
		
		// Go through the the Component list.
		for(int index = 0; index < parent.getComponentCount(); index++) {
			// Retrieve the next component.
			Component component = parent.getComponent(index);
			
			// Check whether the component is visible.
			if(component.isVisible()) {
				// Retrieve the constraint that go with this Component.
				GridSpanConstraints constraints = getConstraints(component);
				
				// Check whether there is not already a component in the grid at the
				// positions taken by this Component.
				if(isFree(componentgrid, constraints.getX(), constraints.getY(), constraints.getWidth(), constraints.getHeight())) {
					 // Put the Component in the portion.
					 setTaken(componentgrid, constraints.getX(), constraints.getY(), constraints.getWidth(), constraints.getHeight(), component);
				}
				// If the potion is not free, check whether the current component should
				// be made invisble.
				else if(active) {
					// Make this Component invisible. It will not be layed out. Note that
					// we don't turn off its visiblity, because then we're messing with
					// the users preferences and also, if the conflicting Components are
					// gone, then this one should show up again.
					// So we just resize it to nothing (^^).
					component.setBounds(0, 0, 0, 0);
				}
			}
		}
		
		// Return the grid.
		return componentgrid;
	}
	
	/**
	 * Checks whether the given portion of the <code>Component</code> grid does not
	 * contain any <code>Component</code>s.
	 * 
	 * @param componentgrid The grid with <code>Component</code>s.
	 * @param x The left corner of the portion.
	 * @param y The upper corner of the portion.
	 * @param width The width of the portion.
	 * @param height The height of the portion.
	 * @return <code>true</code> when the given portion has only nulls,
	 * <code>false</code> otherwise.
	 */
	protected boolean isFree(Component[][] componentgrid, int x, int y, int width, int height) {
		// Go through the rows of the portion.
		for(int row = y; row < y + height; row++) {
			// Go through the columns of the portion.
			for(int column = x; column < x + width; column++) {
				// Check the current cell. If there is a Component return false.
				if(componentgrid[column][row] != null)  return false;
			} 
		}
		
		// If we get here every cell is free. Return true to indicate this.
		return true;
	}

	/**
	 * Set the passed portion of the passed grid to the passed
	 * <code>Component</code>.
	 * 
	 * @param componentgrid The grid with <code>Component</code>s.
	 * @param x The left corner of the portion.
	 * @param y The upper corner of the portion.
	 * @param width The width of the portion.
	 * @param height The height of the portion.
	 * @param component The <code>Component</code> to be set in the postion's cells.
	 */
	protected void setTaken(Component[][] componentgrid, int x, int y, int width, int height, Component component) {
		// Go through the rows of the portion.
		for(int row = y; row < y + height; row++) {
			// Go through the columns of the portion.
			for(int column = x; column < x + width; column++) {
				// Set the current cell to the passed Component.
				componentgrid[column][row] = component;
			} 
		}
	}

	/**
	 * Returns the total width of the passed array with widths, taken into account
	 * only the largest width per column.
	 * 
	 * @param cellwidths The array with the cell widths. <code>null</code> in the
	 * array indicates spanned widths.
	 * @return The total width of this array.
	 */
	protected int getGridWidth(Integer[][] cellwidths) {
		// The width.
		int width = 0;
		
		// Go through the columns.
		for(int column = 0; column < getWidth(); column++) {
			// Add the current column's largest width to the total width.
			width += getLargestWidth(cellwidths, column);
		}	
		
		// Return the total width.
		return width;
	}

	/**
	 * Return the largest width of the all the cells in the indicated column.
	 * This method is recursive, and will take into account the -1 values
	 * (spanned cells).
	 * 
	 * @param cellwidths The array with the cell widths. <code>null</code> in the
	 * array indicates spanned widths.
	 * @param column The column to be searched.
	 * @return The largest width the the passed column.
	 */
	protected int getLargestWidth(Integer[][] cellwidths, int column) {
		// The largest width in this column.
		int largestwidth = 0;
		
		// Go through the rows.
		for(int row = 0; row < getHeight(); row++) {
			// The width of the current cell.
			Integer cellwidth = cellwidths[column][row];
			
			// The comparation width of the current cell.
			// If the cell is a spanned cell it remains 0.
			int comparationwidth = 0;
			
			// Check whether this is not a spanned cell.
			if(cellwidth != null) {
				// This cell has a normal cell width, or is the end of a spanned cell.
				// Make the comparation width of this cell its width minus the
				// largest widths of the previous spanned cells.
				// Start with making it the width of the current cell.
				comparationwidth = cellwidth.intValue();
				
				// The index pointing to the current previous cell.
				int index = column - 1;
				
				// Go through the previous cells until the beginning, or a positive
				// value is found.
				while(index >= 0 && cellwidths[row][index] == null) {
					// Subtract the largest width of previous column from the current
					// cell width.
					comparationwidth -= getLargestWidth(cellwidths, index);
					
					// Decrease index.
					index--;
				}
			}
			
			// Set the largest width to the current cell width if it is larger then
			// the current largest cell width.
			largestwidth = Math.max(comparationwidth, largestwidth);
		}
		
		// Return the largest width.
		return largestwidth;
	}

	/**
	 * Returns the total height of the passed array with heights, taken into account
	 * only the largest height per row.
	 * 
	 * @param cellheights The array with the cell heights. <code>null</code> in the
	 * array indicates spanned heights.
	 * @return The total height of this array.
	 */
	protected int getGridHeight(Integer[][] cellheights) {
		// The height.
		int height = 0;
		
		// Go through the rows.
		for(int row = 0; row < getHeight(); row++) {
			// Add the current row's largest height to the total height.
			height += getLargestHeight(cellheights, row);
		}	
		
		// Return the total height.
		return height;
	}

	/**
	 * Return the largest height of the all the cells in the indicated row.
	 * This method is recursive, and will take into account the -1 values
	 * (spanned cells).
	 * 
	 * @param cellheights The array with the cell heights. <code>null</code> in the
	 * array indicates spanned heights.
	 * @param column The row to be searched.
	 * @return The largest height the the passed row.
	 */
	protected int getLargestHeight(Integer[][] cellheights, int row) {
		// The largest height in this row.
		int largestheight = 0;
		
		// Go through the columns.
		for(int column = 0; column < getWidth(); column++) {
			// The height of the current cell.
			Integer cellheight = cellheights[column][row];
			
			// The comparation height of the current cell.
			// If the cell is a spanned cell it remains 0.
			int comparationheight = 0;
			
			// Check whether this is not a spanned cell.
			if(cellheight != null) {
				// This cell has a normal cell height, or is the end of a spanned
				// cell. Make the comparation height of this cell its height minus
				// the largest heights of the previous spanned cells.
				// Start with making it the width of the current cell.
				comparationheight = cellheight.intValue();
				
				// The index pointing to the current previous cell.
				int index = column - 1;
				
				// Go through the previous cells until the beginning, or a positive
				// value is found.
				while(index >= 0 && cellheights[row][index] == null) {
					// Subtract the largest height of previous column from the
					// current cell height.
					comparationheight -= getLargestHeight(cellheights, index);
					
					// Decrease index.
					index--;
				}
			}
			
			// Set the largest height to the current cell width if it is larger then
			// the current largest height.
			largestheight = Math.max(comparationheight, largestheight);	
		}
		
		// Return the largest height.
		return largestheight;
	}
	
	
	/**
	 * The width of the grid in cells.
	 */
	protected int width;

	/**
	 * The height of the grid in cells.
	 */
	protected int height;
	
	
	/**
	 * The horizontal gap.
	 */
	protected int horizontalgap;

	/**
	 * The vertical gap.
	 */
	protected int verticalgap;
	
	
	/**
	 * The Map that links the components to their constraints objects.
	 */
	protected HashMap componentmap;
}
