/*
 * Copyright 2004 Berend "Kirk" Wouda
 * 
 * This file is part of GPL'd.
 * 
 * GPL'd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * GPL'd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GPL'd; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// The package.
package system;

// Import the AWT stuff we need.
import java.awt.event.*;

// Import the GUI.
import gui.GPLdFrame;

// Import the io classes.
import java.io.*;

// Import FileLineList.
import kirk.io.FileLineList;
import kirk.io.WriteNotAllowedException;

// Import the FolderFilter.
import kirk.io.FolderFilter;

/**
 * This class provides the coupling between the "see" and the "do".
 * 
 * @author Berend "Kirk" Wouda
 * @version 1.00
 * @since 1.00
 */
public class GPLdControl implements ActionListener {
	/**
	 * Constructs a new control class.
	 * 
	 * @param gui The GUI this contorl class is coupled with.
	 */
	public GPLdControl(GPLdFrame gui) {
		// Set the gui.
		this.gui = gui;
	}
	
	
	/**
	 * Gets called when the user interacts with the GUI.
	 * 
	 * @param event The <code>ActionEvent</code> object for this event.
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent event) {
		// Retrieve the action command from the event. 
		String command = event.getActionCommand();
		
		// Find out which button was pressed.
		     // The Start button.
		     if(command.equals("START"))  start();
		     // The folder Browse button.
		else if(command.equals("BROWSEFOLDER"))  gui.setFoldername(gui.askForFolderSelection());
		     // The license Browse button.
		else if(command.equals("BROWSELICENSE"))  gui.setLicenseFilename(gui.askForLicenseFileSelection());
	}
	
	
	/**
	 * Starts the adding process.
	 */
	protected void start() {
		// Catch errors.
		try {
			// Get the folder name and turn it into a file.
			File folder = new File(gui.getFoldername());
			
			// Check whether the file exists.
			if(folder.exists()) {
				// Check if it's a folder.
				if(folder.isDirectory()) {
					// Call GPL.
					GPL(folder);
			
					// All files have been edited.
					gui.popup("All files have been successfully edited.");
				}
				else {
					// Not a folder!
					throw new IOException("The foldername does not specify a folder, but a file: " + folder);
				}
			}
			else {
				// Not a file!
				throw new IOException("The folder does not exist: " + folder);
			}
		}
		// An error has occured. It has to do with the file.
		catch(WriteNotAllowedException e) {
			// Display the error message to the user.
			gui.popup("Writing is not allowed: " + e.getMessage());
		}
		// An error has occured. It has to do with the file.
		catch(IOException e) {
			// Display the error message to the user.
			gui.popup("A file read or write error has occured: " + e.getMessage());
		}
	}
	
	/**
	 * <p>This method does the actual work, it adds the correctly formatted GPL to
	 * the source files. Of course, <code>FileLineList</code> does all the
	 * <i>actual</i> work :^P</p>
	 * 
	 * <p>This method is recursive.</p>
	 * 
	 * <p>Precondition is that the passed <code>File</code> exists, and is a
	 * folder.</p>
	 * 
	 * @param folder The folder to go through.
	 * @throws IOException When IO stuff goes wrong.
	 * @throws WriteNotAllowedException When writing is not allowed.
	 */
	protected void GPL(File folder) throws IOException, WriteNotAllowedException {
		// List all the files with the given extension in this folder.
		String[] filelist = folder.list(new FilenameFilter(){
			public boolean accept(File dir, String filename) {
				// Accept if it's not a folder and if it ends with the given
				// extension.
				return !(new File(dir, filename).isDirectory()) && filename.endsWith("." + gui.getExtension());
			}
		});
		
		// Go through the files.
		for(int fileindex = 0; fileindex < filelist.length; fileindex++) {
			// The current filename:
			String filename = folder + System.getProperty("file.separator") + filelist[fileindex];
			
			// Create a new FileLineList that can write (for editing) on the
			// current file.
			FileLineList linelist = new FileLineList(filename, true);
			
			// The inset.
			String inset = gui.getCommentInset();
			
			// The current line index in the file.
			int lineindex = 0;
			
			// Add the starting comment tag, the copyright notice and a white
			// line to the file.
			linelist.insert(gui.getStartCommentTag(), lineindex++);
			linelist.insert(inset + "Copyright " + gui.getCopyrightYear() + " " + gui.getCopyrightName(), lineindex++);
			linelist.insert(inset, lineindex++);
			
			// Check if the program is a multifile program.
			if(gui.getMultiFileProgram()) {
				// Add the multifile program clause and a whiteline.
				linelist.insert(inset + "This file is part of " + gui.getProgramName() + ".", lineindex++);
				linelist.insert(inset, lineindex++);
			}
			
			// Open up the source GPL license in a reading FileLineList.
			FileLineList gpl = new FileLineList(gui.getLicenseFilename(), false);
			
			// The current index in the GPL file.
			int gplindex;
			
			// Go through the license.
			for(gplindex = 0; gplindex < gpl.getNumberOfLines(); gplindex++) {
				// Retrieve a line, scan it for <name> and replace those
				// instances. Then add it to the sourcefile.
				linelist.insert(inset + gpl.get(gplindex).replaceAll("<name>", gui.getProgramName()), lineindex++);
			}
			
			// Add the end comment tag and 2 whitelines.
			linelist.insert(gui.getEndCommentTag(), lineindex++);
			linelist.insert("", lineindex++);
			linelist.insert("", lineindex++);
			
			// Write the changes to the file.
			linelist.flush(filename);
		}
		
		// Check whether recursive searching is on.
		if(gui.getRecursive()) {
			// List all the folders in the current folder.
			File[] folderlist = folder.listFiles(new FolderFilter());
				
			// Go through the folders, and recurse this method into them.
			for(int folderindex = 0; folderindex < folderlist.length; folderindex++) {
				// Recurse.
				GPL(folderlist[folderindex]);
			}
		}
	}
	
	
	/**
	 * The GUI this control class is coupled with.
	 */
	protected GPLdFrame gui;
}